C+
C NAME:
C	smei_htm_sky
C PURPOSE:
C	Indexes a frame of SMEI data.
C CATEGORY:
C	camer/for/htm
C CALLING SEQUENCE:
	subroutine smei_htm_sky(iframe,hdr,frame)
C INPUTS:
C	iframe		integer		frame counter
C	hdr(*)		double precision frame header
C	frame(*)	real		2-D SMEI frame
C OUTPUTS:
C IMPLICIT:
	implicit none
C INCLUDE:
	include		'filparts.h'
	include		'smei_frm_layout.h'
	include		'smei_frm_hdr.h'
	include		'smei_htm_dim.h'
C CALLS:
C	smei_htm_indexpixel, smei_hdr_quaternion, quaternion_rotate_xyz
C	smei_ccd2cam, smei_hdr_time, smei_orbit2, FitPlane, Say
C	Time2Str, Str2Str, Int2Str, Flt2Str, Dbl2Str
C	iSetFileSpec, iGetFileSpec, iOSDeleteFile, cInt2Str, ForeignArgSet
C	smei_htm_go, iFilePath, itrim, iGetLun, iFreeLun
C	BadI4, BadR4, BadR8, quaternion_inverse
C SEE ALSO:
C	smei_htm_init, smei_htm_pix, smei_htm_lowres, smei_htm_make
C PROCEDURE:
C >	Find the 4 vertices of a pixel, transform the ccd coordinates to sky coordinates,
C	and send the 4 ra/dec vertices to the C++ smei_htm_indexpixel routine.
C >	Uses the hierarchical triangle coordinate system from
C	Johns Hopkins, as a base upon which the photometric measurements
C	from a single SMEI camera are combined and then averaged to produce
C	a surface-brightness sky map. The "level" parameter governs the
C	granularity of the triangular coordinate frame. The coordinates of
C	the four corners of a pixel (or bin) are transformed down into this
C	frame, and the photometric surface brightness as measured by that
C	pixel is added to all nodes whose centers lie within the rectangle.
C	This surface brightness is corrected by 1/cosine of the pixels angle
C	to the camera axis, and by r/rad0 the pixels fractional distance to
C	the FOVs rotational center. When the data sequence is finished, all
C	triangle response sums are averaged and the resulting sky map binned
C	into an output surface brightness map.
C MODIFICATION HISTORY:
C	2002,2003 Aaron Smith (UCSD/CASS)
C	2002->Original
C	2003->removed most C++ functions and re-wrote in fortran as subroutines
C	    ->add quaternion capabilities
C	DEC-2002, Paul Hick (UCSD/CASS)
C	    Defined LEVEL and NMAX as parameters. NMAX is determined from LEVEL.
C	    NMAX is then used to dimension arrays node, nodeid and hits
C	JAN-2003, Andrew Buffington (UCSD/CASS)
C	    Integrated standard pedestal and dark current routines, FORTRAN
C	    Rendered names compliant with those of B.V. Jackson
C	    Incorporated 1/cosine surface-brightness correction for aperture
C	    perspective (currently in smei_htm_sky routine)
C	    Incorporated large and small scale flatfield corrections
C	    Incorporated FORTRAN read subroutine
C	Jan-May, 2003 Aaron Smith (UCSD)
C	  note: not compliant with TMO data anymore
C	FEB-2005, Paul Hick (UCSD/CASS)
C	    Reduced resolution of polar maps from 1600x1600 to 800x800.
C	    Added theta-x and theta-y maps for glare removal as
C	    low-resolution maps to Fits skymap.
C	NOV-2005, Paul Hick (UCSD/CASS)
C	    Added another low resolution map to output file (fraction
C	    of orbit in seconds since start of orbit).
C	    The lowres maps for the 'glare angles' are now always calculated.
C	    They used to be set zero if glare subtraction was suppressed.
C	    All lowres maps for camera 2 are run through GridSphere2D to
C	    fill holes near the equatorial poles.
C	DEC-2005, Paul Hick (UCSD/CASS; pphick@ucsd.edu)
C	    Modified to deal with presence of -overwrite on cmd line
C-
	    integer		iframe
	    double precision	hdr  (*)
	    real		frame(*)

	character	cSaySky   *7		/'htm_sky'   /
	character	cSayInit  *8		/'htm_init'  /
	character	cSayUpdate*10		/'htm_update'/
	character	cSayFts   *7		/'htm_fts'   /

	character	cSeverity		/'E'/

	real		 MAX_SLOPE
	parameter	(MAX_SLOPE = 50.0)

	!================================
	! HTM arrays:
	! Initialized in smei_htm_init.
	! Filled in smei_htm_output
	! Used in smei_htm_make to fill the equatorial and polar skymaps
	
	! Dimensions for node, nodeid, hits above must be at least 2*(4**(level+1))
	! For LEVEL = 12, NMAX = 134217728
	! For LEVEL =  8, NMAX =    524288

	! node	 stores the response value for an HTM node triangle
	! nodeid stores the name of a triangle in format(uint64 in C++) used
	!		in JHU Spatial Index routines
	! hits	 stores the number of frames contributing to a nodes response

	!=================================
	! Frame arrays.
	! Initialized in smei_htm_init.
	! Filled in smei_htm_sky.
	! Written to disk in smei_htm_out.

	integer		badpix(SMEI__FRM_NPIX)
	save		badpix

	real		lsff(SMEI__FRM_NPIX)
	save		lsff

	real		cal_pattern_dark
	save		cal_pattern_dark

	real		cal_pattern(SMEI__FRM_NPIX)
	save		cal_pattern

	real		orb_pattern(SMEI__FRM_NPIX)
	save		orb_pattern

	real		pattern(SMEI__FRM_NPIX)
	save		pattern

	!=================================
	! High-resolution skymap arrays
	! Filled in smei_htm_make from HTM arrays and written to disk in smei_htm_fts

	integer		img_eq(HTM__N_EQ)
	save		img_eq

	real		xmg_eq(HTM__N_EQ)
	save		xmg_eq

	real		row(HTM__NXEQ)		! Scratch array

	integer		mask_eq(HTM__N_EQ)	! Star mask to identify locations close to stars
	save		mask_eq

	byte		jmg_eq(HTM__N_EQ  )	! Used to store node counts in Fits file
	byte		jmg_pl(HTM__N_PL*2)	! Used to store node counts in Fits file
	!character*1	jmg_eq(HTM__N_EQ  )	! Used to store node counts in Fits file
	!character*1	jmg_pl(HTM__N_PL*2)	! Used to store node counts in Fits file
	equivalence	(jmg_eq, jmg_pl)

	integer		img_pl(HTM__N_PL*2)
	save		img_pl

	real		xmg_pl(HTM__N_PL*2)
	save		xmg_pl

	integer		mask_pl(HTM__N_PL*2)	! Star mask to identify locations close to stars
	save		mask_pl

	!=================================
	! Low-resolution skymap arrays
	! String ID for Fits file

	character	cFitsID(HTM__LO_NMAP)*35
     &				/'Dirty sky brightness',
     &				 'Time (fraction of orbit)',
     &				 'PSF position angle from north',
     &				 'PSF position angle from equinox',
     &				 'Discarded response map',
     &				 'Discarded HTM nodes',
     &				 'Theta-x for glare removal',
     &				 'Theta-y for glare removel',
     &				 'Time (sec since first frame used)',
     &				 '',		! Not written to Fits file
     &				 ''/		! Not written to Fits file

	real		lowres(HTM__NXLO,HTM__NYLO,HTM__LO_NMAP)
	save		lowres

	real		lowval(HTM__LO_NMAP)

	!=================================
	! smei_htm_init:

	logical		bGo

	logical		bOverwrite
	save		bOverwrite

	logical		bExists
	save		bExists

	logical		bCheckVersion
	save		bCheckVersion

	integer		iSilent
	save		iSilent

	integer		iType
	save		iType

	logical		bKeepGlitch
	save		bKeepGlitch

	! sets : icam, mode, level, cOut, iOut (saved internally)
	! initializes: 6 low resolution sky arrays
 
	character	cVar(*)*(*)
	character	cArg*(*)

	character	cSkyType*7	/'.fts.gz'/
	save		cSkyType

	integer		thisorbit	/-1/ ! Integer orbit nr for internal use
	save		thisorbit

	character	cSplit*(FIL__LENGTH)! Build from cVar(4) and thisorbit
	save		cSplit		! Template for name of output files

	integer		iSplit		! Loc in cSplit where file name id is inserted
	save		iSplit

	integer		jSplit		! Effective length of template
	save		jSplit

	integer		level		! Extracted from cmd line cArg
	save		level

	integer*2	level2

	integer		icam		! Copy of icam_
	save		icam

	integer		mode		! Copy of mode_
	save		mode

	integer		nbin		! Derived from mode
	save		nbin		! .. binning (1,2,4)

	integer		nX		! Derived from mode
	save		nX		! .. hor frame size (1272,636,318)

	integer		nY		! Derived from mode
	save		nY		! .. vert frame size (256,128,64)

	integer		nfov		! Derived from first SMEI frame
	save		nfov		! .. # pixels inside of SMEI fov

	integer		pfov(HTM__NFOV)	! Filled from first SMEI frame
	save		pfov		! .. linear pixel indices of nfov pixels inside SMEI fov

	real		geometric(HTM__NFOV)
	save		geometric

	double precision runit(3,0:4,HTM__NFOV)
	save		 runit

	real		badvalue
	save		badvalue
					! The biggest glare map is a mode 1 frame, but don't change this!!!
	real		glare_map(SMEI__FRM_NPIX)	/SMEI__FRM_NPIX*0.0/
	save		glare_map

	real		glare_factor(HTM__NX_GLARE,HTM__NY_GLARE)
	save		glare_factor

	real		cx_glare
	save		cx_glare

	real		cy_glare
	save		cy_glare

	real		dx_glare
	save		dx_glare

	real		dy_glare
	save		dy_glare

	double precision ra_min
	save		 ra_min

	double precision ra_max
	save		 ra_max

	double precision dec_min
	save		 dec_min

	double precision dec_max
	save		 dec_max

	integer		tfirstfrm(2)	! Time of first frame used
	save		tfirstfrm

	integer		tlastfrm(2)	! Time of current frame
	save		tlastfrm

	integer		nxeq_
	integer		nyeq_
	integer		n_eq_
	double precision d_eq_
	double precision cxeq_
	double precision cyeq_

	integer		nxpl_(2)
	integer		nypl_(2)
	integer		n_pl_(2)
	double precision d_pl_(2)
	double precision cxpl_(2)
	double precision cypl_(2)

	! These six are accessed only in smei_htm_init

	real		zcut(0:2)	/111.00, 27.76,  6.94/	!500 electron cut

	integer		near_star
	integer		nhot_pix
	real		zz(-1:1,-1:1)
	real		z0
	real		rcx
	real		rcy

	double precision dfov(4)	/ 0.0d0, 0.0d0, 0.0d0, 0.0d0/
	double precision dx(4)		/-0.5d0,-0.5d0, 0.5d0, 0.5d0/	!Lower-left,Upper-left,upper-right,lower-right
	double precision dy(4)		/-0.5d0, 0.5d0, 0.5d0,-0.5d0/

	integer		n_frame_start	/9/

	real		 DRFOV_MAX
	parameter	(DRFOV_MAX = 25.0)

	real		 T_POOL
	parameter	(T_POOL = 36.0)

	integer		n_pool
	save		n_pool

	character	cStr*(FIL__LENGTH)
	character	cTmp*(FIL__LENGTH)
	character	cTime*29

	double precision x(0:4)
	double precision y(0:4)
	double precision dr
	double precision phi
	double precision rr
	double precision tx
	double precision ty
	double precision rx
	double precision ry
	double precision rz
	double precision px
	double precision py
	double precision pz
	double precision dval

	real		rval
	real		darkratio

	integer		i
	integer		j
	integer		k
	integer		ii
	integer		jj
	integer		ibeg
	integer		iend
	integer		n
	integer		nbad
	integer		nneg
	integer		ip
	integer		iU
	integer		istat

	character	cInt2Str*14
	logical		ForeignArgSet
	logical		smei_htm_go
	logical		smei_inside_fov
	integer		smei_htm_pixel2node
	integer		ibset
	integer		Str2Str
	integer		Int2Str
	integer		Flt2Str
	integer		Dbl2Str
	integer		Time2Str
	integer		iFilePath
	integer		itrim
	integer		iGetLun
	integer		iFreeLun
	integer		iSetFileSpec
	integer		iGetFileSpec
	integer		iOSDeleteFile
	integer		iArrI8ValuePresent
	integer		BadI4
	real		BadR4
	double precision BadR8

	double precision dsind				! Needed for GNU compiler
	double precision dcosd
	double precision dasind
	double precision datan2d

	character	cName*(FIL__LENGTH)

	logical		bSimple		/.TRUE./	! Conforms to FITS standard
	integer		iBitPix
	integer		nAxis		/     2/	! Two dim array
	integer		nAxes(2)
	logical		bExtend		/.TRUE./	! Allows FITS extensions

	real		rbad

	double precision qq_cam2equ(4)
	double precision qq_equ2cam(4)

	real		ped
	real		dark

	logical		bOneSky
	save		bOneSky

	logical		bKeepGlare
	save		bKeepGlare

	logical		bInRADecOff
	save		bInRaDecOff

	logical		bInRADec

	logical		bglare
	logical		bNewCal
	logical		bNewOrb
	real		glare_multiplier
	real		response1
	real		response2

	double precision version
	double precision fraction(2)

	integer		time(2)
	integer		hms (4)
	integer		nframe

	integer		ip_pix
	integer		ip_eq
	integer		ip_pl

	ip_pix(i,j  ) = (j-1) *nX  +i				! Used to index SMEI frame arrays
	ip_eq (i,j  ) =	(j-1) *HTM__NXEQ+i			! Used to index equatorial map array
	ip_pl (i,j,k) = ((k-1)*HTM__NYPL+(j-1))*HTM__NXPL+i	! Used to index polar map arrays

	call smei_hdr_quaternion(hdr,1,qq_equ2cam)	! Equatorial -> camera frame
	call quaternion_inverse(qq_equ2cam,qq_cam2equ)	! Camera -> equatorial frame

	ped  = sngl( hdr(SMEI__HDR_PEDESTAL    ) )
	dark = sngl( hdr(SMEI__HDR_DARK_CURRENT) )

	! Set up timing information for current frame:
	! lowval(HTM__LO_TIME) is a fraction of an orbit
	! lowval(HTM__LO_SECS) is the number of seconds since tfirstfrm

	call smei_orbit2(tlastfrm,i,dval)		! Current frame time (set in smei_htm_init)
	lowval(HTM__LO_TIME) = sngl(dval)		! Fraction in current orbit

	call Time2Delta(tlastfrm,tfirstfrm,time)
	call Time2HMS(0,time,hms)			! Seconds since start of orbit
	lowval(HTM__LO_SECS) = float((hms(1)*60+hms(2))*60+hms(3))
							! (hms(4) should always be zero)
	i = 0
	i = i+Str2Str('c' ,cName(i+1:))
	i = i+Int2Str(icam,cName(i+1:))
	i = i+Str2Str('m' ,cName(i+1:))
	i = i+Int2Str(mode,cName(i+1:))
	i = i+Str2Str('_ ',cName(i+1:))
	i = i+Time2Str('SMEI',tlastfrm,cName(i+1:))

	! Set up glare subtraction for this frame

	rx = hdr( SMEI__HDR_SUN+0 )
	ry = hdr( SMEI__HDR_SUN+1 )
	rz = hdr( SMEI__HDR_SUN+2 )

	lowval(HTM__LO_ROTX) = sngl(datan2d(rx,rz))
	lowval(HTM__LO_ROTY) = sngl(datan2d(ry,rz))

	glare_multiplier = 0.0

	if (.not. bKeepGlare .and. rz .gt. dsind(1d0)) then! More than 1 deg above horizon

	    ii = nint(cx_glare+dx_glare*    lowval(HTM__LO_ROTX) )
	    jj = nint(cy_glare+dy_glare*abs(lowval(HTM__LO_ROTY)))
	    if (1 .le. ii .and. ii .le. HTM__NX_GLARE .and. 1 .le. jj .and. jj .le. HTM__NY_GLARE) then
		glare_multiplier = glare_factor(ii,jj)
		write (*,'(A,2F6.1,2I4,F5.1)') cName(:itrim(cName)), lowval(HTM__LO_ROTX), lowval(HTM__LO_ROTY), ii, jj, glare_multiplier
	    end if

	end if

	bglare = glare_multiplier .ne. 0.0

	darkratio = dark/cal_pattern_dark
	nbad = 0
	nneg = 0

	if (iSilent .lt. 3) call Say(cSaySky,'I','<hh:mm:ss>','.#frame '//cName)

	! Remove nodes that have moved outside the FOV from pool 1.

	call smei_htm_flush(iType,bKeepGlitch,iSilent,1,level,icam,mode,qq_equ2cam,DRFOV_MAX,xmg_eq,img_eq,xmg_pl,img_pl,badpix,lowres)

	call smei_htm_sort(1,iSilent)
	if (n_pool .eq. 2) call smei_htm_sort(2,iSilent)

	! n_pool is initialized to 2 at the beginning of the orbit
	! Switch to pool 1 when T_POOL seconds into the orbit.

	if (lowval(HTM__LO_SECS) .ge. T_POOL) n_pool = 1

	call smei_htm_save()

	do n=1,nfov			! Loop over all pixels inside fov

	    ip = pfov(n)		! Linear index
	    lowval(HTM__LO_SKY) = frame(ip)

	    ! Note that response pretty much always should be positive for pixels
	    ! inside the fov, so this test could be redundant.

	    if (lowval(HTM__LO_SKY) .gt. 0.0) then

		i = 1+mod(ip-1,nX)	! Column index
		j = 1+(ip-1)/nX		! Row index

		! Take away pedestal and dark-current-scaled pattern.
		! Apply flat field correction.

		response1 = lowval(HTM__LO_SKY)
		lowval(HTM__LO_SKY) = (lowval(HTM__LO_SKY)-(ped+pattern(ip)*darkratio))/lsff(ip)
		response2 = lowval(HTM__LO_SKY)

		if (bglare) then	! Take out the glare
		    !print *, lowval(HTM__LO_SKY), i,j
		    lowval(HTM__LO_SKY) = lowval(HTM__LO_SKY)-glare_multiplier*glare_map(ip)
		    !print *, lowval(HTM__LO_SKY)
		end if

		! Print a warning if the response is negative at this point. Note that
		! we haven't made the geometric corrections yet, but that involves just a
		! multiplication that can't make the response negative.

		if (iSilent .lt. 1) then
		    if (lowval(HTM__LO_SKY) .lt. 0) then
			nneg = nneg+1
			if (nneg .lt. 20) then
			    ii = 0
			    ii = ii+Str2Str('I<0 ['		,cStr(ii+1:))
			    ii = ii+Int2Str(i			,cStr(ii+1:))
			    ii = ii+Str2Str(','			,cStr(ii+1:))
			    ii = ii+Int2Str(j			,cStr(ii+1:))
			    ii = ii+Str2Str(']:'		,cStr(ii+1:))+1
			    ii = ii+Flt2Str(response1,-2	,cStr(ii+1:))+1
			    ii = ii+Flt2Str(response2,-2	,cStr(ii+1:))+1
			    ii = ii+Flt2Str(lowval(HTM__LO_SKY)	,-2,cStr(ii+1:))
			else if (nneg .eq. 20) then
			    cStr = '.... and more'
			else
			    cStr = ' '
			end if
			if (cStr .ne. ' ') call Say(cSaySky,'W',cName,cStr)
		    end if
		end if
							! Apply geometric corrections
		lowval(HTM__LO_SKY) = lowval(HTM__LO_SKY)*geometric(n)
		lowval(HTM__LO_HIT) = 1.0

		rx = runit(1,0,n)
		ry = runit(2,0,n)
		rz = runit(3,0,n)

		dval = dsqrt(1d0-rx*rx)

		px =  0d0					! Unit vector for PSF orientation
		py =  rz/dval					! .. in camera coordinate frame
		pz = -ry/dval

		! Camera to equatorial frame

		call quaternion_rotate_xyz(qq_cam2equ,rx,ry,rz)
		call quaternion_rotate_xyz(qq_cam2equ,px,py,pz)

		! psfn and psfe are 'position angles' counterclockwise from the reference
		! direction (as viewed from outside the unit sphere).

		lowval(HTM__LO_PSFN) = sngl( datan2d( ry*px-rx*py,pz) )
		lowval(HTM__LO_PSFE) = sngl( datan2d(-rz*px+rx*pz,py) )

		k = 0						! Center of pixel

		x(k) = datan2d(ry,rx)				! Right ascension
		x(k) = dmod(x(k)+360d0,360d0)			! Force into [0,360)

		rz   = max(-1d0,min(rz,1d0))			! Safety belt for dasind fnc
		y(k) = dasind(rz)				! Declination

		! Check whether pixel is in selected piece of sky

		bInRADec = bInRADecOff

		if (.not. bInRADecOff) then
		    bInRADec = dec_min .le. y(k) .and. y(k) .le. dec_max
		    if (bInRADec) bInRADec =
     &			(ra_min .le. ra_max .and. (ra_min .le. x(k) .and. x(k) .le. ra_max)) .or.
     &			(ra_min .gt. ra_max .and. (ra_min .le. x(k) .or.  x(k) .le. ra_max))
		end if

		if (bInRADec) then

		    do k=1,4					! Corners (lower-right, upper-left, upper-right, lower-right) 
			rx = runit(1,k,n)
			ry = runit(2,k,n)
			rz = runit(3,k,n)
								! Camera --> equatorial frame
			call quaternion_rotate_xyz(qq_cam2equ,rx,ry,rz)

			x(k) = datan2d(ry,rx)			! Right ascension
			x(k) = dmod(x(k)+360d0,360d0)		! Force into [0,360)

			rz   = max(-1d0,min(rz,1d0))		! Safety belt for dasind fnc
			y(k) = dasind(rz)			! Declination
		    end do					! end vertex loop

		    ! Before sending the pixel to the indexing routine determine whether
		    ! or not it is close to a bright star.

		    jj = nint(HTM__CYEQ+y(0)*HTM__D_EQ)

		    if (1 .le. jj .and. jj .le. HTM__NYEQ) then	! Latitude inside equatorial map

			ii = nint(HTM__CXEQ+x(0)*HTM__D_EQ)	! (ii,jj) is nearest bin in equat map

			if (ii .lt. 1 .or. ii .gt. HTM__NXEQ) then! Better not happen!
			    write (*,*) ' RA=',x(0),' Dec=',y(0)
			    stop 'smei_htm_sky: bad RA, darn'
			end if

			ii = ip_eq(ii,jj)
			near_star = mask_eq(ii)			! 0=away from star; 1=near star

		    else					! Use polar map

			! The equatorial map and polar maps overlap, so if a point is not in the
			! equatorial map, it must be in one of the polar maps.

			dval = HTM__D_PL*(90d0-dabs(y(0)))

			ii = nint(HTM__CXPL+dval*dcosd(x(0)))
			jj = nint(HTM__CYPL+dval*dsind(x(0)))	! (ii,jj) is nearest bin in polar map

			if (ii .lt. 1 .or. ii .gt. HTM__NXPL .or. jj .lt. 1 .or. jj .gt. HTM__NYPL) then
			    write (*,*) ' RA=',x(0),' Dec=',y(0)! Better not happen!
			    stop 'smei_htm_sky: bad dec, darn'
			end if

			ii = ip_pl(ii,jj,(3-sign(1,int(y(0))))/2)! 1 for North, 2 for South
			near_star = mask_pl(ii)			! 0=away from star; 1=near star

		    end if

		    nhot_pix = 0

		    if (icam .eq. 3 .or. near_star .eq. 1) then

			do jj=-1,1
			    do ii=-1,1
				k = ip_pix(i+ii,j+jj)
				zz(ii,jj) = (frame(k)-(ped+pattern(k)*darkratio))/lsff(k)
			    end do
			end do

			call FitPlane(zz,zz,z0,rcx,rcy,rval)

			! If close to bright star, then check slope. If it is less than
			! the threshold slope MAX_SLOPE, switch off the near_star flag

			if (near_star .eq. 1) then
			   if (sqrt(rcx*rcx+rcy*rcy) .le. MAX_SLOPE) near_star = 0
			end if

			if (icam .eq. 3) then

			    if (abs(zz(0,0)) .gt. zcut(mode)) then

				!if((z(0,0)*z(0,0))/rval.gt.5.)then
				!write (*,'(A,F8.2,A,F10.2,2(A,F7.2),A,F10.2,2(A,I3),A)
     				!&	'response = ',lowval(HTM__LO_SKY),'  sumzq = ',rval,'  zz(0,0) = ',zz(0,0),'  z0 = ',z0,
     				!&	'  zz(0,0)**2/sumzsqr = ',zz(0,0)*zz(0,0)/rval,'  (i,j) = (',i,',',j,')'
				!endif

				! Remove dark current slope from hot/flipper pixel flag

				if (zz(0,0)*zz(0,0) .gt. 25.0*rval .or. (pattern(ip)-cal_pattern_dark*0.4*(1.0+0.01*j)
     &				    .gt. 50.0 .and. zz(0,0) .lt. 0.0)) nhot_pix = 1

			    end if

			end if

		    end if

		    ! All four corners of the pixel are used: x(1..4),y(1..4)

		    ii = 0
		    if (near_star .eq. 1) ii = ibset(ii,0)
		    if (nhot_pix  .eq. 1) ii = ibset(ii,1)
		    call smei_htm_fill(level,mode,n_pool,x(1),y(1),ip,lowval(HTM__LO_SKY),ii)

		    ! Determine which low-res bin to drop the current pixel.

		    !   0 <= x <  360 -> 1 <= ii <= NXLO  , i.e. ii is always valid
		    ! -90 <= y <= +90 -> 1 <= jj <= NYLO+1, i.e. jj is invalid for y=+90

		    ii = nint(HTM__CXLO+x(0)*HTM__D_LO)		! Pixel center
		    jj = nint(HTM__CYLO+y(0)*HTM__D_LO)
		    jj = min(jj,HTM__NYLO)			! Only used if y(0) exactly +90d0

		    if (ii .lt. 1 .or. ii .gt. HTM__NXLO .or. jj .lt. 1 .or. jj .gt. HTM__NYLO) then
			write (*,*) ' RA=',x(0),' Dec=',y(0)	! Better not happen!
			stop 'htm_sky: bad low res bin, darn'
		    end if

		    ! Sum low resolution maps

		    call smei_htm_addangle(lowval(HTM__LO_PSFN),lowres(ii,jj,HTM__LO_HIT),lowres(ii,jj,HTM__LO_PSFN))! PSF rotation
		    call smei_htm_addangle(lowval(HTM__LO_PSFE),lowres(ii,jj,HTM__LO_HIT),lowres(ii,jj,HTM__LO_PSFE))
		    call smei_htm_addangle(lowval(HTM__LO_ROTX),lowres(ii,jj,HTM__LO_HIT),lowres(ii,jj,HTM__LO_ROTX))! Glare angles
		    call smei_htm_addangle(lowval(HTM__LO_ROTY),lowres(ii,jj,HTM__LO_HIT),lowres(ii,jj,HTM__LO_ROTY))
		    call smei_htm_addtime (lowval(HTM__LO_TIME),   0.5,lowres(ii,jj,HTM__LO_HIT),lowres(ii,jj,HTM__LO_TIME))! Fractions
		    call smei_htm_addtime (lowval(HTM__LO_SECS),3048.0,lowres(ii,jj,HTM__LO_HIT),lowres(ii,jj,HTM__LO_SECS))! Seconds

		    lowres(ii,jj,HTM__LO_SKY) = lowres(ii,jj,HTM__LO_SKY)+lowval(HTM__LO_SKY)	! Sidereal skymap at low resolution
		    lowres(ii,jj,HTM__LO_HIT) = lowres(ii,jj,HTM__LO_HIT)+lowval(HTM__LO_HIT)

		end if

	    else

		nbad = nbad+1

	    end if

	end do

	if (iSilent .lt. 2) then

	    call smei_htm_print()

	    if (nneg .ne. 0) then
		i = 0
		i = i+Str2Str('intensity in', cStr(i+1:))+1
		i = i+Int2Str(nneg	, cStr(i+1:))
		i = i+Str2Str('/'	, cStr(i+1:))
		i = i+Int2Str(nfov	, cStr(i+1:))
		i = i+Str2Str(' pixels'	, cStr(i+1:))
		call Say(cSaySky,'W','negative',cStr)
	    end if

	    if (nbad .ne. 0) then
		i = 0
		i = i+Int2Str(nbad	, cStr(i+1:))
		i = i+Str2Str('/'	, cStr(i+1:))
		i = i+Int2Str(nfov	, cStr(i+1:))
		i = i+Str2Str(' bad pixels'   , cStr(i+1:))
		call Say(cSaySky,'W','ignored',cStr)
	    end if

	end if

	return
C+
C NAME:
C	smei_htm_init
C PURPOSE:
C	Initialize variables for smei_htm_sky.
C CALLING SEQUENCE:
	entry smei_htm_init(version,hdr,cVar,cArg,bGo)
C INPUTS:
C	version		double precision	version number
C	hdr(*)		double precision	frame header array
C	cVar(*)		character*(*)		cVar(2): pattern file or SMEIDB?
C						cVar(4): output directory
C	cArg		character*(*)		command line keywords
C OUTPUTS:
C	bGo		logical			bGo is returned .FALSE. only
C						if the first frame for a new skymap
C						is processed, and the Fits file for
C						the skymap already exists.
C CALLS:
C	ForeignI4Arg, smei_orbit2, smei_orbit_time2, Str2Str, Int2Str, Time2Str, iFilePath
C	ArrR4Zero, ArrI8Zero, ArrI2Zero, ArrI4Zero, Say, cInt2Str, smei_cal_get
C	smei_get_glare, ForeignArgSet, smei_hdr_time
C PROCEDURE:
C	Entry point in href=smei_htm_sky=.
C MODIFICATION HISTORY:
C	NOV-2004, Paul Hick (UCSD/CASS)
C	DEC-2005, Paul Hick (UCSD/CASS; pphick@ucsd.edu)
C	    Modified to deal with presence of -overwrite on cmd line
C-
	bGo = .TRUE.

	call smei_hdr_time(hdr,tlastfrm)	! Frame time, saved for Fits header

	if (thisorbit .eq. -1) then		! First frame of first orbit
						! If bOneSky is set then this section is
						! .. only passed once.
	    bOneSky = ForeignArgSet(cArg,'onesky')

	    ! First frame of first orbit: do stuff that needs to be done only once

	    icam  = nint(hdr(SMEI__HDR_CAMERA))	! Save: camera
	    mode  = nint(hdr(SMEI__HDR_MODE  ))	! Save: mode
	    nbin  = 2**mode			! Save: binning

	    call ForeignI4Arg(cArg,'level',-1,level)

	    if (level .eq. -1) then		! No level specified
	    	level = 11			! Science modes @ level 11
		if (mode .eq. 0) level = 12	! Engineering mode @ level 12
	    end if

	    if (level .gt. HTM__LEVEL) call Say(cSayInit,'E','level '//cInt2Str(level),
     &		'is too high; max is HTM__LEVEL='//cInt2Str(HTM__LEVEL))

	    level = max(1,min(level,HTM__LEVEL))! Save: level

	    call ForeignI4Arg(cArg,'silent',0,iSilent)
	    bKeepGlare  = ForeignArgSet(cArg,'keepglare' )
	    bKeepGlitch	= ForeignArgSet(cArg,'keepglitch')
	    if (ForeignArgSet(cArg,'mean')) then
		iType = 0
	    else if (ForeignArgSet(cArg,'median')) then
		iType = 1
	    else
		call ForeignI4Arg(cArg,'type',0,iType)
	    end if

	    ! Limit construction of skymap to box of sky in RA and dec

	    call ForeignR8Arg(cArg,'ra_min' , -1.0, ra_min )
	    call ForeignR8Arg(cArg,'ra_max' , -1.0, ra_max )
	    call ForeignR8Arg(cArg,'dec_min',-91.0, dec_min)
	    call ForeignR8Arg(cArg,'dec_max',-91.0, dec_max)

	    ! All four must be set. If one is omitted the others are silently ignored

	    bInRADecOff = ra_min  .lt.   0.0 .or. ra_max  .lt.   0.0 .or.
     &			  dec_min .lt. -90.0 .or. dec_max .lt. -90.0

	    if (.not. bInRADecOff) then

		! Adjust input values making sure they are located at pixel
		! edges (at half-integer array index values.

		ra_min = mod(mod(ra_min,360.0)+360.0,360.0)
		ra_max = mod(mod(ra_max,360.0)+360.0,360.0)

		ra_min = (nint(HTM__CXEQ+ra_min*HTM__D_EQ)-0.5-HTM__CXEQ)/HTM__D_EQ
		ra_max = (nint(HTM__CXEQ+ra_max*HTM__D_EQ)+0.5-HTM__CXEQ)/HTM__D_EQ

		dec_min = (nint(HTM__CYEQ+dec_min*HTM__D_EQ)-0.5-HTM__CYEQ)/HTM__D_EQ
		dec_max = (nint(HTM__CYEQ+dec_max*HTM__D_EQ)+0.5-HTM__CYEQ)/HTM__D_EQ

		dec_min = max(-90.0,min(dec_min,90.0))
		dec_max = max(-90.0,min(dec_max,90.0))

		i = 0
		i = i+Str2Str('RA range (deg):'	 , cStr(i+1:))+1
		i = i+Dbl2Str(ra_min, 3		 , cStr(i+1:))
		i = i+Str2Str(' -'		 , cStr(i+1:))+1
		i = i+Dbl2Str(ra_max, 3		 , cStr(i+1:))
		i = i+Str2Str(', Dec range (deg)', cStr(i+1:))
		i = i+Dbl2Str(dec_min, 3	 , cStr(i+1:))
		i = i+Str2Str(' -'		 , cStr(i+1:))+1
		i = i+Dbl2Str(dec_max, 3	 , cStr(i+1:))

		call Say(cSayInit,'E','Limit',cStr)

	    end if

	    if (bKeepGlare) then

		call Say(cSayInit,'I','glare','subtraction skipped')

	    else

		bKeepGlare =  icam .eq. 1

		if (bKeepGlare) then

		    call Say(cSayInit,'I','glare','subtraction only for c2 and c3')

		else

		    ! glare_map    is a 318x 64 array for c2m2 and 636x128 for c3m1
		    ! glare_factor is a  90x180 array

		    call smei_get_glare(icam,mode,glare_map,glare_factor,cx_glare,cy_glare,dx_glare,dy_glare)

		end if

	    end if

	    badvalue = 0.0			! Save: bad value flag (put in Fits header)

	    nX = SMEI__FRM_NX/nbin		! Save
	    nY = SMEI__FRM_NY/nbin		! Save

	    ! Get the star masks (0 = away from star; 1 is near star)

	    call smei_htm_starmask(HTM__N_EQ,mask_eq,HTM__N_PL,mask_pl)

	    ! Get large-scale flatfield correction. Zeroes are set to one.

	    call smei_get_lsff(mode,icam,lsff)

	    ! Run through the first frame to make a list of all pixels inside the fov
	    ! that need to be processed.

	    ibeg = (SMEI__FRM_LEFT-1)/nbin+1	! Partially covered column
	    iend =  SMEI__FRM_RIGHT  /nbin

	    if (mode .eq. 0) then		! For mode 0 drop the half-covered column on either side
		ibeg = ibeg+1
		iend = iend-1
	    end if

	    nfov = 0

	    do j=1,nY

		do i=ibeg,iend

		    rx = dble(i)
		    ry = dble(j)

		    ! Convert from ccd coordinates to sky coordinates in the camera
		    ! coordinate frame (with z along the optical axis and x along the long
		    ! dimension of the fov).

		    call smei_axis_ccd(2,icam,mode,rx,ry,dr,phi,rr)

		    ! dr : radial distance from optical axis (in mode 0 pixels)
		    ! phi: angle from optical axis
		    ! rr : (radial distance from origin)/(radial dist of opt axis)

		    if (smei_inside_fov(icam,mode,dfov,dr,phi)) then	! Pixel inside FOV ?

			k = ip_pix(i,j)		! Linear array index

			! Exclude pixels along the edge. For mode 1 and 2 these could
			! be bins including the partially covered pixels on left or right.
			! For camera 3 this also avoids problems when filling the
			! zz(-1:1,-1:1) array. None of these should be inside the fov,
			! so the check is probably redundant.

			if (i .ne. ibeg .and. i .ne. iend .and. j .ne. 1 .and. j .ne. nY) then
			    nfov = nfov+1	! Save: number of pixels inside SMEI fov
			    if (nfov .gt. HTM__NFOV) call Say(cSayInit,'E','HTM__NFOV='//cInt2Str(HTM__NFOV),'is too small')

			    pfov(nfov) = k	! Save: 1-based linear array index into SMEI frame

			    call smei_axis_cam(0,icam,mode,dr,phi,runit(1,0,nfov),runit(2,0,nfov),runit(3,0,nfov),tx,ty)

			    ! Geometric corrections to response using pixel center position.
			    ! - First order radial binsize correction
			    ! - Radial binsize, nonlinear term
			    ! - Correct surface brightness for aperture effective area

			    geometric(nfov) = sngl(rr*(1d0-0.0092d0*ty)/runit(3,0,nfov))

			    ! Unit vectors in camera frame for all four corners of the pixel

			    do n=1,4
				call smei_axis_ccd(0,icam,mode,rx+dx(n),ry+dy(n),dr,phi,rr)
				call smei_axis_cam(0,icam,mode,dr,phi,runit(1,n,nfov),runit(2,n,nfov),runit(3,n,nfov),tx,ty)
			    end do

			else
			    k = 0
			    k = k+Str2Str('[', cStr(k+1:))
			    k = k+Int2Str( i , cStr(k+1:))
			    k = k+Str2Str(',', cStr(k+1:))
			    k = k+Int2Str( j , cStr(k+1:))
			    k = k+Str2Str(']', cStr(k+1:))
			    call Say(cSayInit,'W',cStr,'ditched edge pixel inside fov')

			end if

		    end if

		end do

	    end do

	    call Say(cSayInit,'I',cInt2Str(nfov),'pixels inside the FOV')

	    ! Build file name template for output files.
	    ! cVar(4) stores the output directory.

	    if (bOneSky) then

		! Store time of first good frame
		! (used in file name construction, is stored in Fits header
		! and is used as time origin for the orbit time (in secs)

		call ArrI4Copy(2,tlastfrm,tfirstfrm)

		i = 0
		i = i+Str2Str ('c' , cStr(i+1:))
		i = i+Int2Str (icam, cStr(i+1:))
		i = i+Str2Str ('_' , cStr(i+1:))
		iSplit = i			! Save
		i = i+Time2Str('SMEI',tfirstfrm, cStr(i+1:))
						! cVar(4) is output directory
		jSplit = iFilePath(cVar(4),0,'',cStr,cSplit) !Save
		iSplit = iSplit+jSplit-i	! Save

		! bOverwrite: .TRUE. if -overwrite is set on command line
		! bCheckversion: .TRUE. if -checkversion is set on command line
		! bGo	 : signals to main program whether to keep processing frames
		! bExists: used in smei_frm_fts to delete a file if -overwrite is set

		bOverwrite    = ForeignArgSet(cArg,'overwrite')
		bCheckVersion = ForeignArgSet(cArg,'checkversion')

		cStr = cSplit(:iSplit-1)//'sky'//cSplit(iSplit:jSplit)//cSkyType

		bGo = smei_htm_go(bOverwrite,bCheckVersion,cStr,version,bExists)

		n_pool = 2

		call ArrI4Zero(HTM__N_EQ  ,img_eq)
		call ArrR4Zero(HTM__N_EQ  ,xmg_eq)

		call ArrI4Zero(HTM__N_PL*2,img_pl)
		call ArrR4Zero(HTM__N_PL*2,xmg_pl)

		call ArrI4Zero(SMEI__FRM_NPIX/(nbin*nbin),badpix)

		call ArrR4Zero(HTM__N_LO*HTM__LO_NMAP,lowres)! Initialize low resolution maps

		i = 0
		i = i+Time2Str('SMEI',tfirstfrm,cStr(i+1:))
		i = i+Str2Str(', camera', cStr(i+1:))+1
		i = i+Int2Str(icam	, cStr(i+1:))
		i = i+Str2Str(', mode'	, cStr(i+1:))+1
		i = i+Int2Str(mode	, cStr(i+1:))+1
		i = i+Str2Str('('	, cStr(i+1:))
		i = i+Int2Str(nbin	, cStr(i+1:))
		i = i+Str2Str('x'	, cStr(i+1:))
		i = i+Int2Str(nbin	, cStr(i+1:))+1
		i = i+Str2Str('binned)'	, cStr(i+1:))+1
		i = i+Str2Str('@ level'	, cStr(i+1:))+1
		i = i+Int2Str(level	, cStr(i+1:))

		call Say(cSayInit,'I','start',cStr)

		call smei_orbit2(tfirstfrm,thisorbit,dval)

	    end if

	end if

	if (.not. bOneSky) then

	    call smei_orbit2(tlastfrm,i,dval)

	    if (i .ne. thisorbit) then		! First frame of new orbit

		! First frame of new orbit: do stuff that needs to be done
		! for each orbit.

		thisorbit = i			! New orbit number
		call ArrI4Copy(2,tlastfrm,tfirstfrm)
						! Orbit start time to nearest second
		call smei_orbit_time2(thisorbit,0d0,time)
		time(2) = nint(dble(time(2))/1000)*1000

		! Build file name template for output files.
		! cVar(4) stores the output directory.

		i = 0
		i = i+Str2Str ('c' , cStr(i+1:))
		i = i+Int2Str (icam, cStr(i+1:))
		i = i+Str2Str ('_' , cStr(i+1:))
		iSplit = i			! Save
		i = i+Time2Str('SMEI',time,cStr(i+1:))
						! cVar(4) is output directory
		jSplit = iFilePath(cVar(4),0,'',cStr,cSplit) !Save
		iSplit = iSplit+jSplit-i	! Save

		! bOverwrite: .TRUE. if -overwrite is set on command line
		! bCheckversion: .TRUE. if -checkversion is set on command line
		! bGo	 : signals to main program whether to keep processing frames
		! bExists: used in smei_frm_fts to delete a file if -overwrite is set

		bOverwrite    = ForeignArgSet(cArg,'overwrite')
		bCheckVersion = ForeignArgSet(cArg,'checkversion')

		cStr = cSplit(:iSplit-1)//'sky'//cSplit(iSplit:jSplit)//cSkyType

		bGo = smei_htm_go(bOverwrite,bCheckVersion,cStr,version,bExists)

		n_pool = 2

		call ArrI4Zero(HTM__N_EQ  ,img_eq)
		call ArrR4Zero(HTM__N_EQ  ,xmg_eq)

		call ArrI4Zero(HTM__N_PL*2,img_pl)
		call ArrR4Zero(HTM__N_PL*2,xmg_pl)

		call ArrI4Zero(SMEI__FRM_NPIX/(nbin*nbin),badpix)

		call ArrR4Zero(HTM__N_LO*HTM__LO_NMAP,lowres)! Initialize low resolution maps

		i = 0
		i = i+Str2Str('camera'  , cStr(i+1:))+1
		i = i+Int2Str(icam	, cStr(i+1:))+1
		i = i+Str2Str('in mode' , cStr(i+1:))+1
		i = i+Int2Str(mode	, cStr(i+1:))+1
		i = i+Str2Str('('	, cStr(i+1:))
		i = i+Int2Str(nbin	, cStr(i+1:))
		i = i+Str2Str('x'	, cStr(i+1:))
		i = i+Int2Str(nbin	, cStr(i+1:))+1
		i = i+Str2Str('binned)'	, cStr(i+1:))+1
		i = i+Str2Str('@ level' , cStr(i+1:))+1
		i = i+Int2Str(level	, cStr(i+1:))

		call Say(cSayInit,'I','orbit '//cInt2Str(thisorbit),cStr)

	    end if

	end if

	! Finally, do stuff that needs to be done for each frame.

	! The calibration pattern is either stored in cVar(2) or if cVar(2) is 'SMEIDB?'
	! it is selected from the SMEI database based on the time stored in 'tlastfrm'.
	! For camera 3 in mode 1 cVar(5) contains the orbital difference pattern to be
	! added to the calibration pattern.

	if (icam .eq. 3 .and. mode .eq. 1) then

	    call smei_cal_get(cVar(2),icam,mode,tlastfrm,cStr,cal_pattern,cal_pattern_dark,bNewCal)
	    call smei_orb_get(cVar(5),icam,mode,cStr,orb_pattern,bNewOrb)

	    if (bNewCal .or. bNewOrb) call ArrR4PlusArrR4(nX*nY,cal_pattern,orb_pattern,pattern)

	else

	    call smei_cal_get(cVar(2),icam,mode,tlastfrm,cStr,pattern,cal_pattern_dark,bNewCal)

	end if

	if (nint(hdr(SMEI__HDR_CAMERA)) .ne. icam .or.
     &	    nint(hdr(SMEI__HDR_MODE  )) .ne. mode)
     &		call Say(cSayInit,'E','wrong','camera or mode')

	return
C+
C NAME:
C	smei_htm_make
C PURPOSE:
C	Takes the three HTM node arrays (node, nodeid, and hits),
C	averages them and puts them into a two-dimensional skymap
C CATEGORY:
C	camera/for/htm
C CALLING SEQUENCE:
	entry smei_htm_make()
C OUTPUTS:
C	Output goes to three equatorial .grd files, two polar .grd files
C	and a FTS file containing both equatorial and polar maps.
C CALLS:
C	Say, ArrI4Zero, ArrR4Zero, TRIANGLECENTER, smei_htm_node
C	ArrR4DivideByArrI4, ArrR4DivideByArrR4, ArrR4Mask
C PROCEDURE:
C >	Entry point in href=smei_htm_sky=. See that procedure for more information.
C MODIFICATION HISTORY:
C	April 2003, Aaron Smith (UCSD/CASS)
C	    Updated to include capability for all sky imaging in 3 equatorial
C	    plots, and two polar plots
C-

	! Put all remaining nodes on the skymaps

	rval = BadR4()
	call smei_htm_flush(iType,bKeepGlitch,iSilent,1,level,icam,mode,qq_equ2cam,rval,xmg_eq,img_eq,xmg_pl,img_pl,badpix,lowres)
	call smei_htm_flush(iType,bKeepGlitch,iSilent,2,level,icam,mode,qq_equ2cam,rval,xmg_eq,img_eq,xmg_pl,img_pl,badpix,lowres)

	call smei_htm_sort(1,iSilent)	! Not really necessary (clears n_nodes entries)
	call smei_htm_sort(2,iSilent)

	call smei_htm_print_nodes(mode)

	! The division of xmg and img will set all bins with img=0 to BadR4().
	! Set the bad values to zero before writing to file.

	call ArrR4DivideByArrI4(HTM__N_EQ  ,xmg_eq,img_eq,xmg_eq)
	call ArrR4DivideByArrI4(HTM__N_PL*2,xmg_pl,img_pl,xmg_pl)

	! All low resolution maps

	call ArrR4DivideByArrR4(HTM__N_LO,lowres(1,1,HTM__LO_SKY ),lowres(1,1,HTM__LO_HIT),lowres(1,1,HTM__LO_SKY ))
	call ArrR4DivideByArrR4(HTM__N_LO,lowres(1,1,HTM__LO_TIME),lowres(1,1,HTM__LO_HIT),lowres(1,1,HTM__LO_TIME))
	call ArrR4DivideByArrR4(HTM__N_LO,lowres(1,1,HTM__LO_SECS),lowres(1,1,HTM__LO_HIT),lowres(1,1,HTM__LO_SECS))

	call ArrR4DivideByArrR4(HTM__N_LO,lowres(1,1,HTM__LO_PSFN),lowres(1,1,HTM__LO_HIT),lowres(1,1,HTM__LO_PSFN))
	call ArrR4DivideByArrR4(HTM__N_LO,lowres(1,1,HTM__LO_PSFE),lowres(1,1,HTM__LO_HIT),lowres(1,1,HTM__LO_PSFE))

	call ArrR4DivideByArrR4(HTM__N_LO,lowres(1,1,HTM__LO_ROTX),lowres(1,1,HTM__LO_HIT),lowres(1,1,HTM__LO_ROTX))
	call ArrR4DivideByArrR4(HTM__N_LO,lowres(1,1,HTM__LO_ROTY),lowres(1,1,HTM__LO_HIT),lowres(1,1,HTM__LO_ROTY))

	call ArrR4DivideByArrR4(HTM__N_LO,lowres(1,1,HTM__LO_CUT ),lowres(1,1,HTM__LO_CNT),lowres(1,1,HTM__LO_CUT))
	call ArrR4DivideByArrR4(HTM__N_LO,lowres(1,1,HTM__LO_FRAC),lowres(1,1,HTM__LO_CNT),lowres(1,1,HTM__LO_FRAC))

	! The low res maps have small areas near the poles which do not get filled,
	! i.e. lowres(*,*,HTM__LO_HIT) remains zero. The GridSphere3D calls are should fill these.
	! (the DO loop can be avoided, but that would increase the required size of the
	! scratch arrays in GridSphere3D)
	! Only do this for camera 2 (the other two don't get close enough to the pole
	! to cause problems).

	if (icam .eq. 2) then
	    i    = 2+10			! Fill holes, use open grid
	    rval = 360.0/HTM__NXLO	! Gaussian halfwidth in degrees
	    !call GridSphere2D(1.0,HTM__NXLO,HTM__NYLO,1,lowres(1,1,HTM__LO_SKY ),rval,i	,0.0,0.0)
	    !call GridSphere2D(1.0,HTM__NXLO,HTM__NYLO,1,lowres(1,1,HTM__LO_TIME),rval,i	,0.0,0.0)
	    !call GridSphere2D(1.0,HTM__NXLO,HTM__NYLO,1,lowres(1,1,HTM__LO_SECS),rval,i	,0.0,0.0)
	    call GridSphere2D(1.0,HTM__NXLO,HTM__NYLO,1,lowres(1,1,HTM__LO_PSFN),rval,i+1000	,0.0,0.0)
	    call GridSphere2D(1.0,HTM__NXLO,HTM__NYLO,1,lowres(1,1,HTM__LO_PSFE),rval,i+1000	,0.0,0.0)
	    call GridSphere2D(1.0,HTM__NXLO,HTM__NYLO,1,lowres(1,1,HTM__LO_ROTX),rval,i+1000	,0.0,0.0)
	    call GridSphere2D(1.0,HTM__NXLO,HTM__NYLO,1,lowres(1,1,HTM__LO_ROTY),rval,i+1000	,0.0,0.0)
	    !call GridSphere2D(1.0,HTM__NXLO,HTM__NYLO,1,lowres(1,1,HTM__LO_FRAC),rval,i	,0.0,0.0)
	    !call GridSphere2D(1.0,HTM__NXLO,HTM__NYLO,1,lowres(1,1,HTM__LO_CUT ),rval,i	,0.0,0.0)
	end if

	! Ugly, but no way to avoid this, I think: set BadR4() values to zero.

	rbad = BadR4()

	call ArrR4Mask(HTM__N_EQ  ,xmg_eq,rbad,0.0,0.0,0.0,1.0,xmg_eq)
	call ArrR4Mask(HTM__N_PL*2,xmg_pl,rbad,0.0,0.0,0.0,1.0,xmg_pl)

	call ArrR4Mask(HTM__N_LO*HTM__LO_NMAP,lowres,rbad,0.0,0.0,0.0,1.0,lowres)

	! The rotation angles could be slightly outside the range (-180,+180].
	! Make sure they are strictly inside this range.
	! Note that this has to be done AFTER BadR4 has been replaced by 0.0.

	do j=1,HTM__NYLO
	    do i=1,HTM__NXLO
		call smei_htm_fixangle(lowres(i,j,HTM__LO_PSFN))
		call smei_htm_fixangle(lowres(i,j,HTM__LO_PSFE))
		call smei_htm_fixangle(lowres(i,j,HTM__LO_ROTX))
		call smei_htm_fixangle(lowres(i,j,HTM__LO_ROTY))
	    end do
	end do

	return
C+
C NAME:
C	smei_htm_fts
C PURPOSE:
C	Writes orbital sky amps to FTS file
C CATEGORY:
C	ucsd/camera/for/htm
C CALLING SEQUENCE:
	entry smei_htm_fts(version,fraction,nframe,iframe)
C CALLS:
C	iGetLun, iFreeLun, ArrR4Mask, Say, say_fts
C	iSetFileSpec, iGetFileSpec, Time2Split, itrim
C	Time2System, Time2Str, smei_orbit_time2, smei_orbit_period2
C	FTINIT, FTPHPR, FTPKYS, FTPKYJ, FTPKYE, FTPKYD, FTPKYG
C	FTPPRE, FTCRHD, FTCLOS, FTPPRB, FTPPRJ, iOSDeleteFile
C PROCEDURE:
C >	Entry point in href=smei_htm_sky=.
C
C >	The equatorial maps use a linear scale to from array index i,j to ra,dec
C		i=1,nxeq, ra=[0,360), j=1,nyeq, dec=[-60,60)
C		i = cxeq+ra *d_eq	ra  = (i-cxeq)/d_eq
C		j = cyeq+dec*d_eq	dec = (j-cyeq)/d_eq
C >	The polar maps use a polar transformation
C		i=1,nxeq, j=1,nyeq
C		i = cxpl+d_pl*(90-abs(dec))*cos(ra)
C		j = cypl+d_pl*(90-abs(dec))*sin(ra)
C		ra  = atan( (j-cypl)/(i-cxpl) )
C		90-abs(dec) = sqrt( (i-cxpl)^2+(j-cypl)^2 )
C >	The equatorial map is written as primary data, the north and
C	south polar maps as first and second extension, respectively.
C >	Added entries to FTS header:
C
C	Ext 0 (equatorial map):
C		SMEI_HTM	character	Software version number
C		CREATED		character	File creation time
C		NAME		character	File name (no directory, no type)
C		CAMERA		integer		Camera  
C		ORBIT		double precision Orbit number
C		MAP		character	String describing map
C		CXEQ		double precision Array index for RA=0 
C		CYEQ		double precision Array index for DEC=0
C		D_EQ		double precision Number of bins per degree
C
C	Ext 1 (north pole map)
C		MAP		character	String describing map
C		CXPL		double precision Array index for north pole
C		CYPL		double precision Array index for north pole
C		D_PL		double precision Number of bins per degree
C
C	Ext 2 (south pole map)
C		MAP		character	String describing map
C		CXPL		double precision Array index for south pole
C		CYPL		double precision Array index for south pole
C		D_PL		double precision Number of bins per degree
C
C	Note that the values for both poles are the same.
C
C >	After the high-resolution skymaps follow a group of (currently) six
C	low resolution maps:
C
C	Ext 3-8 ('dirty' sky brightness, time, psf orientation w.r.t. equatorial
C	north and w.r.t. vernal equinox, fraction of discarded response, and
C	fraction of bad htm nodes)
C		MAP		character	String describing map
C		CXLO		double precision Array index for south pole
C		CYLO		double precision Array index for south pole
C		D_LO		double precision Number of bins per degree
C MODIFICATION HISTORY:
C	NOV-2004, Paul Hick (UCSD/CASS)
C	APR-2005, Paul Hick (UCSD/CASS)
C	    New set of camera quaternions for cam 1 and 2.
C	    Ad hoc corrections to tx and ty in smei_ccd2cam.
C	    Added 'onesky' keyword.
C	    Extra keyword ONEORBIT to identify orbital maps.
C	NOV-2005, Paul Hick (UCSD/CASS)
C	    Modified to accepts different glare maps for cameras 2 and 3.
C	DEC-2005, Paul Hick (UCSD/CASS; pphick@ucsd.edu)
C	    Modified to deal with presence of -overwrite on cmd line.
C	    Added Fits keyword MODE.
C	    Added processing of SMEI_HTM keyword for existing sky maps
C	    to force overwrite of maps with lower version number.
C-
	if (bInRADecOff) then

	    d_eq_ = HTM__D_EQ
	    n_eq_ = HTM__N_EQ
	    nxeq_ = HTM__NXEQ
	    nyeq_ = HTM__NYEQ
	    cxeq_ = HTM__CXEQ
	    cyeq_ = HTM__CYEQ

	    do i=1,2
		d_pl_(i) = HTM__D_PL
		n_pl_(i) = HTM__N_PL
		nxpl_(i) = HTM__NXPL
		nypl_(i) = HTM__NYPL
		cxpl_(i) = HTM__CXPL
		cypl_(i) = HTM__CYPL
	    end do

	else

	    call smei_htm_eqscan(ra_min,ra_max,dec_min,dec_max,
     &		HTM__D_EQ ,HTM__N_EQ ,HTM__NXEQ ,HTM__NYEQ ,HTM__CXEQ ,HTM__CYEQ ,
     &		     d_eq_,     n_eq_,     nxeq_,     nyeq_,     cxeq_,     cyeq_,	xmg_eq,img_eq,row)
	    call smei_htm_plscan(ra_min,ra_max,dec_min,dec_max,
     &		HTM__D_PL ,HTM__N_PL ,HTM__NXPL ,HTM__NYPL ,HTM__CXPL ,HTM__CYPL ,
     &		     d_pl_,     n_pl_,     nxpl_,     nypl_,     cxpl_,     cypl_,	xmg_pl,img_pl)

	end if

	cStr = cSplit(:iSplit-1)//'sky'//cSplit(iSplit:jSplit)//cSkyType

	! Delete the file if it exists already

	if (bExists) then
	    if (.not. bOverwrite .and. .not. bCheckVersion) call Say(cSayFts,'E','#'//cStr,'exists. Should not have happened')
	    i = iOSDeleteFile(cStr)
	    call Say(cSayFts,'I','#'//cStr,'deleted')
	end if

	! Write FITS file

	iU = iGetLun(cStr)

	istat = 0	! Initialize to zero
			! Stays zero as long as there are no FITS errors

	! Open a new Fits file. This will fail if the Fits file
	! exists already.

	call Say(cSayFts,'I','#'//cStr,'is skymap file')
	call FTINIT(iU, cStr, 1, istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	! Write equatorial map as primary data

	nAxes(1) = nxeq_
	nAxes(2) = nyeq_
	iBitPix  =  -32

	! Write primary header

	call FTPHPR(iU, bSimple, iBitPix, nAxis, nAxes, 0, 1, bExtend, istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	! Add to primary header

							! Software version number indicator
	call FTPKYG(iU, 'SMEI_HTM', version, 2, 'Software version number', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call Time2System(time)				! File creation time
	i = Time2Str('SMEI',time,cName)

	call FTPKYS(iU, 'CREATED', cName(:i), 'File creation time', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	i = iSetFileSpec(cStr)
	i = iGetFileSpec(FIL__NAME,FIL__NAME,cName)
							! File name
	call FTPKYS(iU, 'NAME', cName(:i), 'File name', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPKYS(iU, 'IMG_TYPE', 'skymap_htm', 'Image type', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPKYJ(iU, 'CAMERA', icam, 'Camera', istat)! Camera  
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPKYJ(iU, 'MODE', mode, 'Mode', istat)	! Mode
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	i = i+Time2Str('SMEI', tfirstfrm, cTime)
	call FTPKYS(iU, 'STIME', cTime, 'First frame used', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	i = i+Time2Str('SMEI', tlastfrm, cTime)
	call FTPKYS(iU, 'ETIME', cTime, 'Last frame used', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPKYJ(iU, 'NFRAMES', nframe, 'Total nr frames', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPKYJ(iU, 'IFRAMES', iframe, 'Nr frames used', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPKYL(iU, 'ONEORBIT', .not. bOneSky, 'Orbital map', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	if (.not. bOneSky) then
							! Orbit number
	    call FTPKYJ(iU, 'ORBIT', thisorbit, 'Orbit number', istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    call FTPKYG(iU, 'LOWFRAC', fraction(1), 7, 'Start orbital fraction', istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    call FTPKYG(iU, 'HIGHFRA', fraction(2), 7, 'End orbital fraction', istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    call smei_orbit_time2(thisorbit,0d0,time)	! Exact orbit start time
	    i = i+Time2Str('SMEI.fff', time, cTime)
	    call FTPKYS(iU, 'TIME', cTime, 'Orbit start time', istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    call smei_orbit_period2(thisorbit,0d0,dval)	! Exact orbit period
	    call FTPKYG(iU, 'PORBIT', dval*86400d0, 3, 'Orbit period in seconds', istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	end if
							! Map descriptor
	call FTPKYS(iU, 'MAP', 'Equatorial map, RA=[0,360],DEC=[-60,+60]', 'Map description', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPKYG(iU, 'CXEQ', cxeq_, 2, 'Array index (base 1) for RA=0', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPKYG(iU, 'CYEQ', cyeq_, 2, 'Array index (base 1) for DEC=0', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPKYG(iU, 'D_EQ', d_eq_, 2, 'Number of bins per degree', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPKYE(iU, 'BAD_DATA', badvalue, 5, 'Missing data flag', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPPRE(iU, 0, 1, n_eq_, xmg_eq, istat)	! Array
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)


	! Write polar maps as first and second extension

	iBitPix = -32

	do i=1,2

	    nAxes(1) = nxpl_(i)
	    nAxes(2) = nypl_(i)

	    call FTCRHD(iU, istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    call FTPHPR(iU, bSimple, iBitPix, nAxis, nAxes, 0, 1, bExtend, istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    ! Scaling constants for north pole map

	    if (i .eq. 1) then
		call FTPKYS(iU, 'MAP', 'Map of north pole, DEC=[+50,+90]', 'Map description', istat)
	    else
		call FTPKYS(iU, 'MAP', 'Map of south pole, DEC=[-50,-90]', 'Map description', istat)
	    end if
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    call FTPKYG(iU, 'CXPL', cxpl_(i), 2, 'Array index (base 1) for north pole', istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    call FTPKYG(iU, 'CYPL', cypl_(i), 2, 'Array index (base 1) for north pole', istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    call FTPKYG(iU, 'D_PL', d_pl_(i), 2, 'Number of bins per degree', istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)


	    call FTPKYE(iU, 'BAD_DATA', badvalue, 5, 'Missing data flag', istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    call FTPPRE(iU, 0, 1, n_pl_(i), xmg_pl((i-1)*n_pl_(1)+1), istat)	! Write array
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	end do





	do i=1,n_eq_
	    jmg_eq(i) = img_eq(i)
	end do



	nAxes(1) = nxeq_
	nAxes(2) = nyeq_
	iBitPix = 8

	call FTCRHD(iU, istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPHPR(iU, bSimple, iBitPix, nAxis, nAxes, 0, 1, bExtend, istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPKYS(iU, 'MAP', 'Counts for equatorial map, RA=[0,360],DEC=[-60,+60]', 'Map description', istat)

	call FTPKYG(iU, 'CXEQ', cxeq_, 2, 'Array index (base 1) for RA=0', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPKYG(iU, 'CYEQ', cyeq_, 2, 'Array index (base 1) for DEC=0', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPKYG(iU, 'D_EQ', d_eq_, 2, 'Number of bins per degree', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPPRB(iU, 0, 1, n_eq_, jmg_eq, istat)		! Array
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)




	! Write polar maps as first and second extension

	do i=1,n_pl_(1)+n_pl_(2)
	    jmg_pl(i) = img_pl(i)
	end do

	iBitPix = 8

	do i=1,2

	    nAxes(1) = nxpl_(i)
	    nAxes(2) = nypl_(i)

	    call FTCRHD(iU, istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    call FTPHPR(iU, bSimple, iBitPix, nAxis, nAxes, 0, 1, bExtend, istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    ! Scaling constants for north pole map

	    if (i .eq. 1) then
		call FTPKYS(iU, 'MAP', 'Counts for map of north pole, DEC=[+50,+90]', 'Map description', istat)
	    else
		call FTPKYS(iU, 'MAP', 'Counts for map of south pole, DEC=[-50,-90]', 'Map description', istat)
	    end if
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    call FTPKYG(iU, 'CXPL', cxpl_(i), 2, 'Array index (base 1) for north pole', istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    call FTPKYG(iU, 'CYPL', cypl_(i), 2, 'Array index (base 1) for north pole', istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    call FTPKYG(iU, 'D_PL', d_pl_(i), 2, 'Number of bins per degree', istat)
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	    call FTPPRB(iU, 0, 1, n_pl_(i), jmg_pl((i-1)*n_pl_(1)+1), istat)	! Write array
	    if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	end do




	nAxes(1) = HTM__NXLO				! Write low resolution maps
	nAxes(2) = HTM__NYLO
	iBitPix  = -32

	do i=1,HTM__LO_NMAP

	    j = itrim(cFitsID(i))

	    if (j .gt. 0) then
		call FTCRHD(iU, istat)
		if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

		call FTPHPR(iU, bSimple, iBitPix, nAxis, nAxes, 0, 1, bExtend, istat)
		if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

		call FTPKYS(iU, 'MAP', cFitsID(i)(:j)//' RA=[0,360],DEC=[-90,+90]', 'Map description', istat)
		if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

		! Scaling constants

		call FTPKYG(iU, 'CXLO', HTM__CXLO, 2, 'Array index (base 1) for RA=0', istat)
		if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

		call FTPKYG(iU, 'CYLO', HTM__CYLO, 2, 'Array index (base 1) for DEC=0', istat)
		if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

		call FTPKYG(iU, 'D_LO', HTM__D_LO, 2, 'Number of bins per degree', istat)
		if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

		call FTPPRE(iU, 0, 1, HTM__N_LO, lowres(1,1,i), istat)	! Write array
		if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)
	    end if

	end do


	nAxes(1) = nX						! Write bad pixel map
	nAxes(2) = nY
	iBitPix  = 32

	call FTCRHD(iU, istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPHPR(iU, bSimple, iBitPix, nAxis, nAxes, 0, 1, bExtend, istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPKYS(iU, 'MAP', 'Bad pixel map', 'Map description', istat)
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	call FTPPRJ(iU, 0, 1, nX*nY, badpix, istat)		! Write array
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)


	call FTCLOS(iU, istat)					! Close Fits file
	if (istat .ne. 0) call say_fts(cSayFts,cSeverity,istat)

	iU = iFreeLun(iU)

	return
	end
C+
C NAME:
C	smei_htm_addtime
C PURPOSE:
C	Add times for individual frames contributing to sky bin.
C CALLING SEQUENCE:
	subroutine smei_htm_addtime(time,htime,hit,timesum)
C INPUTS:
C	time		real		frame time (orbit fraction
C					or seconds since start of orbit)
C	htime		real		approximately half an orbit
C	hit		real		number of hits in sky bin sofar
C	timesum		real		accumulated times
C OUTPUTS:
C	timesum		real		updated accumulated times
C PROCEDURE:
C	All contributing time values are added together here. The
C	average after dividing by hit will be written to disk.
C
C	At the end of an orbit pixels may contribute to bins where
C	pixels at the beginning of the orbit also contributed. Here values
C	close to 0 and close to one full orbit are combined to produce a
C	funny "mean" time. These bins are flagged as negative.
C MODIFICATION HISTORY:
C	NOV-2005, Paul Hick (UCSD/CASS; pphick@ucsd.edu)
C-
	    real	time
	    real	htime
	    real	hit
	    real	timesum

	if (hit .eq. 0.0) then

	    timesum = time

	else

	    if (timesum .lt. 0.0) then
		timesum =  timesum-time		! Stays negative
	    else if (time-timesum/hit .gt. htime) then
		timesum = -timesum-time		! Switch from positive to negative
	    else
		timesum =  timesum+time		! Still positive
	    end if

	end if

	return
	end
C+
C NAME:
C	smei_htm_addangle
C PURPOSE:
C	Make sure that the rotation angles added together in LO_PSFN and LO_PSFE
C	are all about the same size (i.e. do not add 0.5 and 359.5 but 0.5 and -0.5).
C CALLING SEQUENCE:
	subroutine smei_htm_addangle(angle,hit,anglesum)
C INPUTS:
C	angle		real		angle
C	hit		real		number of hits in bin so far
C	anglesum	real		accumulated angle
C OUTPUTS:
C	anglesum	real		update accumulate angle
C MODIFICATION HISTORY:
C	NOV-2005, Paul Hick (UCSD/CASS; pphick@ucsd.edu)
C-
	    real	angle
	    real	hit
	    real	anglesum

	real		x
	real		d

	if (hit .eq. 0.0) then

	    anglesum = angle

	else

	    x = angle
	    d = x-anglesum/hit
	    if (abs(d) .gt. 180.0) x = x-sign(360.0,d)

	    anglesum = anglesum+x

	end if

	return
	end
C+
C NAME:
C	smei_htm_fixangle
C PURPOSE:
C	Make sure angle is in (-180,+180] degrees
C CALLING SEQUENCE:
	subroutine smei_htm_fixangle(angle)
C INPUTS:
C	angle		real
C OUTPUTS:
C	angle		real
C MODIFICATION HISTORY:
C	NOV-2005, Paul Hick (UCSD/CASS; pphick@ucsd.edu)
C-
	    real	angle

	if (angle .le. -180.0) then
	    angle = angle+360.0
	else if (angle .gt. 180.0) then
	    angle = angle-360.0
	end if

	return
	end
