C+
C NAME:
C	smei_htm_fill
C PURPOSE:
C	Each pixel contributes to several HTM nodes. These 'votes' are added
C	here to the pool of votes for all nodes.
C CATEGORY:
C	smei/camera/for/htm
C CALLING SEQUENCE:
	subroutine smei_htm_fill(level,mode,n_pool,ra_pix,dec_pix,ipos,response,nflag)
C INPUTS:
C	level		integer			determines resolution of pool of nodes
C						(1 <= level <= HTM__LEVEL=3
C	mode		integer			mode 0,1 or 2
C	n_pool		integer			pool in which to drop new nodes in the frame
C	ra_pix(4)	double precision	RA  the four corners of a pixel
C	dec_pix(4)	double precision	dec for four corners of a pixel
C
C	ipos		integer			
C	response	real			CCD response for pixel
C	nflag		integer
C	level		integer
C OUTPUTS:
C CALLS:
C	Say
C INCLUDE:
	include		'smei_htm_dim.h'
C PROCEDURE:
C	The pool of nodes is maintaind in the node_* arrays.
C
C	The nodes in the first frame are all put in pool 2 (with one vote for
C	each node). New nodes in subsequent frames are put in pool 2. If a node
C	already is present in pool 1 or 2 then another vote is for that node.
C
C	Nodes from pool 2 are removed if they move far enough outside the FOV.
C	After the last frame has been processed (and the skymap is complete),
C	both pools are cleared.
C
C	The node_sort array contains the indices needed to sort the node_* arrays
C	containing nodes up to and including contributions from the previous frame,
C	i.e. NOT YET the contribution from the frame currently being processed.
C	At the start node_sort(*) = n_nodes(HTM__NODE_NTOP,*).
C	As pixels from the new frame are added the node_* arrays and
C	n_nodes(HTM__NODE_NTOP,*) are updated; node_sort will be synchronized with
C	n_nodes(HTM__NODE_NTOP,*) by htm_node_sort prior to starting the next frame.
C
C	The implicit assumption is made that within a frame all triangles will have
C	different node names, i.e. NO NODE IS DUPLICATED WITHIN A FRAME.
C MODIFICATION HISTORY:
C	JAN-2006, Paul Hick (UCSD/CASS; pphick@ucsd.edu)
C-
	    integer		level
	    integer		mode
	    integer		n_pool
	    double precision	ra_pix (*)
	    double precision	dec_pix(*)
	    integer		ipos
	    real		response
	    integer		nflag

	!========================
	! Args for smei_htm_flush

	integer		iType
	logical		bKeepGlitch
	integer		nr_pool
	integer		icam
	double precision qq(*)
	real		drfov_max
	integer		img_eq(*)
	real		xmg_eq(*)
	integer		img_pl(*)
	real		xmg_pl(*)
	integer		badpix(*)
	real		lowres(HTM__NXLO,HTM__NYLO,HTM__LO_NMAP)

	!========================
	! Args for smei_htm_sort

	integer		iSilent

	!========================

	integer		n_frame_votes
	save		n_frame_votes

	integer*8	node_list(HTM__NODES_PER_PIXEL)

	integer		n_nodes(HTM__NODE_N,HTM__POOL_N)	/HTM__POOL_S*0/
	save		n_nodes

	integer		n_old_nodes(HTM__NODE_N,HTM__POOL_N)
	save		n_old_nodes

	integer		node_sort (HTM__NODES,HTM__POOL_N)
	save		node_sort

	integer*8	node_name (HTM__NODES,HTM__POOL_N)	! Node ID
	save		node_name

	integer		node_cnt  (HTM__NODES,HTM__POOL_N)	! Counter for votes in node
	save		node_cnt

	integer		node_alive(HTM__NODES,HTM__POOL_N)	! Done with node ?
	save		node_alive

	double precision node_ra  (HTM__NODES,HTM__POOL_N)	! Node right ascension
	save		 node_ra

	double precision node_dec (HTM__NODES,HTM__POOL_N)	! Node declination
	save		 node_dec

	real		node_value(HTM__VOTES,HTM__POOL_N)
	save		node_value

	!integer	node_pos  (HTM__VOTES,HTM__POOL_N)
	!save		node_pos

	integer*1	node_flag (HTM__VOTES,HTM__POOL_N)
	save		node_flag


	!=====================

	integer		max_votes_mode(0:2)		/10,100,150/
	save		max_votes_mode

	double precision px
	double precision py
	double precision rx
	double precision ry
	double precision rz
	double precision tx
	double precision ty
	double precision dr
	double precision phi
	double precision r8bad

	double precision dcosd
	double precision dsind

	character	cInt2Str*14
	integer		Str2Str
	integer		smei_htm_pixel
	integer		smei_htm_combine
	double precision BadR8

	character	cStr*256
	logical		node_not_found
	logical		b_node_alive
	integer*8	node_id
	integer*8	node_newid
	integer		count
	integer*2	level2

	max_votes = max_votes_mode(mode)
	max_nodes = min(HTM__VOTES/max_votes,HTM__NODES)

	level2 = level
	count = smei_htm_pixel(level2,ra_pix,dec_pix,HTM__NODES_PER_PIXEL,node_list)
	if (count .eq. 0) stop 'no votes for pixel?'

	if (count .gt. HTM__NODES_PER_PIXEL)
     &	    call Say('htm_fill','E','HTM__NODES_PER_PIXEL='//cInt2Str(HTM__NODES_PER_PIXEL),
     &		'is too small#need at least '//cInt2Str(count))

	n_frame_votes = n_frame_votes+count

	do i=1,count					! Loop over all nodes in node_list

	    node_newid = node_list(i)			! Node ID of new node
	    if (node_newid .le. HTM__NO_NODE) stop 'smei_htm_node_pool, dang'

	    node_not_found = .TRUE.
	    ipool = n_pool				! First pool to search

	    ! Check whether the node_newid is already in the pool.
	    ! For the first frame n_pool=2 is still empty, so all nodes will end up
	    ! in the second pool. For subsequent frames first pool 1, then pool 2
	    ! is checked to find the node. If it is not found than it is added to
	    ! pool 1.

	    ! All nodes are accessed using the node_sort array, which sorts the
	    ! node is into ascending order).

	    do while (ipool .le. 2 .and. node_not_found)

		node_not_found = n_nodes(HTM__NODE_NSORT,ipool) .eq. 0

		if (.not. node_not_found) then		! No nodes yet? Then bail: node not found

		    ! Lowest and highest node ID
		    ! 1..n_nodes(HTM__NODE_NEMPTY,ipool) were empty at the start of the frame, but
		    ! will be filled by new nodes in the current frame.

		    node_low  = n_nodes(HTM__NODE_NEMPTY,ipool)+1
		    node_high = n_nodes(HTM__NODE_NSORT ,ipool)

		    inode = node_sort(node_low,ipool)	! Check node with lowest ID
		    node_id = node_name(inode,ipool)

		    node_not_found = node_newid .lt. node_id

		    if (.not. node_not_found) then	! Is new ID is lower than lowest ID?
		    					! Then bail: node not found
			node_not_found = node_newid .ne. node_id

			if (node_not_found) then	! Is new ID same as lowest ID? Then bail: node found
							! Only one node? Then bail: node not found
			    node_not_found = node_high .eq. node_low
							! At least two nodes
			    if (.not. node_not_found) then
							! Check node with highest ID
				inode = node_sort(node_high,ipool)
				node_id = node_name(inode,ipool)

				node_not_found = node_newid .gt. node_id
							! Is new ID is higher than highest ID?
							! Then bail: node not found
				if (.not. node_not_found) then

				    node_not_found = node_newid .ne. node_id
							! Is new ID same as highest ID?
							! Then bail: node found
				    if (node_not_found) then

					! Neither node_low nor node_high are right node

					! node_high must be at least two higher than node_low or node_mid
					! will be equal to node_low	
									
					do while (node_not_found .and. node_high-node_low .ge. 2)

					    node_mid = (node_low+node_high)/2

					    inode = node_sort(node_mid,ipool)
					    node_id = node_name(inode,ipool)

					    node_not_found = node_newid .ne. node_id
							! Is new ID same as mid ID? Then bail: node found
					    if (node_not_found) then

						! node_mid is not the right node. Adjust node_high or
						! node_low such that node_low and node_high bracket
						! the node (if it exists), but neither node_low nor
						! node_high is the right node.

						if (node_id .gt. node_newid) then
						    node_high = node_mid
						else
						    node_low  = node_mid
						end if
					    end if

					end do

				    end if

				end if

			    end if

			end if

		    end if

		end if

		if (node_not_found) ipool = ipool+1		! Next pool

	    end do

	    ! If node_not_found=.FALSE. then node_newid is present at (ipool,inode).
	    ! If node_not_found=.TRUE. then find a location for the new node.

	    if (node_not_found) then				! New node

		ipool = n_pool					! Pool 2 for 1st frame; pool 1 after that

		inode = n_nodes(HTM__NODE_IEMPTY,ipool)

		if (inode .lt. n_nodes(HTM__NODE_NEMPTY,ipool)) then	! Empty slot available?

		    inode = inode+1
		    n_nodes(HTM__NODE_IEMPTY,ipool) = inode	! Update number of empty slots filled again

		    inode = node_sort(inode,ipool)		! Empty slot

		    node_id = node_name(inode,ipool)		! Make sure it really is empty
		    if (node_id .ne. HTM__NO_NODE) stop 'node not empty, node_name'

		else						! No empty slots available

		    inode = n_nodes(HTM__NODE_NTOP,ipool)+1	! Add to end of pool

		    if (inode .gt. max_nodes)			! Still room?
     &			call Say('htm_fill','E','max_nodes='//cInt2Str(max_nodes),'is too small')

		    n_nodes(HTM__NODE_NTOP,ipool) = inode	! Update highest occupied index

		end if
								! Update total number of nodes
		n_nodes(HTM__NODE_NSLOT,ipool) = n_nodes(HTM__NODE_NSLOT,ipool)+1

		node_name(inode,ipool) = node_newid		! Store node ID
		node_cnt (inode,ipool) = 0			! Clear counter for votes

		level2 = level					! Store RA and dec
		call TriangleCenter(level2,node_newid,node_ra(inode,ipool),node_dec(inode,ipool))

	    end if

	    if (node_name(inode,ipool) .ne. node_newid) stop 'node_name problem'

	    ! Nodes are still 'alive' as long as votes are being added.

	    node_alive(inode,ipool) = 1

	    ivote = node_cnt(inode,ipool)+1			! Next vote for inode
	    if (ivote .gt. max_votes) then
		print *, ipool,inode,ivote,mode,node_cnt(inode,ipool)
		call WhatIsI4(n_nodes(HTM__NODE_NTOP,ipool),node_cnt(1,ipool),'pool '//cInt2Str(ipool))
		call Say('htm_fill','E','max_votes='//cInt2Str(max_votes),'is too small')
	    end if
	    node_cnt(inode,ipool) = ivote			! Update counter for votes for inode

	    ivote = (inode-1)*max_votes+ivote
	    node_value(ivote,ipool) = response			! CCD readout
	    node_flag (ivote,ipool) = nflag			! Bitwise flags
	    !node_pos (ivote,ipool) = ipos			! Linear array index in frame
								! Update total number of votes in pool
	    n_nodes(HTM__NODE_NVOTE,ipool) = n_nodes(HTM__NODE_NVOTE,ipool)+1

	end do

	return
C+
C NAME:
C	smei_htm_sort
C PURPOSE:
C	Sorts the node ID of all nodes
C CALLING SEQUENCE:
	entry smei_htm_sort(nr_pool,iSilent)
C INPUTS:
C	nr_pool
C OUTPUTS:
C PROCEDURE:
C	The sorted array are described by a couple of arrays
C	that are saved internally
C
C	node_sort		index array to sort nodes
C	n_node_sort(2)		highest index in node_sort used.
C				Is set to value in n_nodes(HTM__NODE_NTOP,ipool)
C	n_nodes(HTM__NODE_NEMPTY,*)
C				number of empty slots in pool
C				(these are located at the bottom of the node_sort array)
C	n_nodes(HTM__NODE_IEMPTY,*)
C				counter used when filling up empty slots in smei_htm_fill
C				(initialized to zero here)
C
C	n_nodes(*,*) is set zero at the start of a new orbit, forcing the
C	initialization of n_nodes(HTM__NODE_NSORT,*), n_nodes(HTM__NODE_NEMPTY,*) and
C	n_nodes(HTM__NODE_IEMPTY,*) to zero.
C MODIFICATION HISTORY:
C	FEB-2006, Paul Hick (UCSD/CASS; pphick@ucsd.edu)
C-
	ipool = nr_pool
	ntop  = n_nodes(HTM__NODE_NTOP,ipool)	! Highest occupied node index

	if (ntop .eq. 0) then			! No nodes in pool

	    n = 0

	else

	    call IndexI8(1,ntop,1,ntop,node_name(1,ipool),node_sort(1,ipool))

	    ! The sorting operation puts available empty slots between 1 and ntop
	    ! at the bottom of the sorted array. Count number of empty slots.

	    n = 0
	    node_id = HTM__NO_NODE				! Redundant

	    do while (n .lt. ntop .and. node_id .eq. HTM__NO_NODE)
		n = n+1
		node_id = node_name(node_sort(n,ipool),ipool)
	    end do

	    if (node_id .eq. HTM__NO_NODE) stop 'bottom problem'

	    n = n-1

	    ! Empty slots are marked as 'alive' to avoid accessing them in smei_htm_flush.

	    do i=1,n
	    	node_alive(node_sort(i,ipool),ipool) = 1	! Empty slots: 'alive'
	    end do

	    ! Occopied slots are marked as 'not alive'. If a vote is added in
	    ! smei_htm_fill by the next frame they will be marked as 'alive' again.
	    ! Only nodes marked as 'not alive' are candidates for removal in
	    ! smei_htm_flush.

	    do i=n+1,ntop
	    	node_alive(node_sort(i,ipool),ipool) = 0	! Filled slots: 'not alive'
	    end do

	end if

	if (iSilent .lt. 2) call Say('htm_sort','I',cInt2Str(n),'empty slots in pool '//cInt2Str(ipool))

	n_nodes(HTM__NODE_NEMPTY,ipool) = n
	n_nodes(HTM__NODE_IEMPTY,ipool) = 0
	n_nodes(HTM__NODE_NSORT ,ipool) = ntop

	return
C+
C NAME:
C	smei_htm_flush
C PURPOSE:
C	Removes node that have moved outside the field of view from the
C	pool of nodes after depositing in on the equatorial and/or polar skymaps
C CALLING SEQUENCE:
	entry smei_htm_flush(iType,bKeepGlitch,iSilent,nr_pool,level,icam,mode,qq,drfov_max,xmg_eq,img_eq,xmg_pl,img_pl,badpix,lowres)
C INPUTS:
C	iType		integer
C	bKeepGlitch	logical
C	nr_pool		integer
C	drfov_max	real
C OUTPUTS:
C PROCEDURE:
C	n_nodes(*,2)	integer		n_nodes(HTM__NODE_NTOP,2)
C					  highest occupied index in pool arrays
C					  updated to a lower value when empty slots
C					  at the end of the pool are removed.
C					n_nodes(HTM__NODE_NSLOT,ipool)
C					  total number of nodes in pool
C					  updated when nodes are removed
C					n_nodes(HTM__NODE_NVOTE,ipool)
C					  total number of votes in pool
C					  update when nodes are removed
C	smei_htm_flush is called in smei_htm_sky prior to adding in the pixels
C	of the a new frame into the pool, i.e. the quaternion from the new
C	header is used to decided whether a node is inside/outside the fov.
C MODIFICATION HISTORY:
C	FEB-2006, Paul Hick (UCSD/CASS; pphick@ucsd.edu)
C-
	ipool = nr_pool
	ntop  = n_nodes(HTM__NODE_NTOP,ipool)	! Highest occupied node

	n_nodes(HTM__NODE_NTOPMAX,ipool) = max(n_nodes(HTM__NODE_NTOPMAX,ipool),n_nodes(HTM__NODE_NTOP,ipool))

	if (ntop .gt. 0) then

	    call ArrI4Copy(HTM__POOL_S,n_nodes,n_old_nodes)

	    ! Sorting node_alive puts nodes that marked 'not alive' (node_alive=0)
	    ! at the bottom of the sorted array. Only these nodes are candidates
	    ! for removal.

	    call IndexI4(1,ntop,1,ntop,node_alive(1,ipool),node_sort(1,ipool))

	    r4bad = BadR4()
	    r8bad = BadR8()
	    max_votes = max_votes_mode(mode)

	    ! The while loop starts at the bottom and the array and continues until a
	    ! a node marked 'alive' (node_alive=1) is encountered.
	    ! If drfov_max is set to r4bad then all nodes are removed (this is used
	    ! to clean up after the last frame for the skymap).

	    n = 1						! Bottom of pool
	    inode = node_sort(n,ipool)				! Sorted index at bottom of pool

	    do while (n .le. ntop .and. (node_alive(inode,ipool) .eq. 0 .or. drfov_max .eq. r4bad))

		! node_name=HTM__NO_NODE only happens if drfov_max=r4bad
		! (i.e. the final cleanup of all nodes).

		node_id = node_name(inode,ipool)

		if (node_id .ne. HTM__NO_NODE) then

		    px = node_ra (inode,ipool)			! Pick up RA and dec
		    py = node_dec(inode,ipool)

		    ! If drfov_max=r4bad then b_node_alive=.FALSE., and the node is always removed

		    b_node_alive = drfov_max .ne. r4bad

		    if (b_node_alive) then

			rx = dcosd(py)*dcosd(px)		! Unit vector in equatorial frame
			ry = dcosd(py)*dsind(px)
			rz = dsind(py)

			call quaternion_rotate_xyz(qq,rx,ry,rz)	! Equatorial --> camera frame
								! Camera frame --> CCD coordinates
			call smei_axis_cam(1,icam,mode,dr,phi,rx,ry,rz,tx,ty)

			! Node must be more than drfov_max pixels
			! dr = r8bad occurs after a data gap

			b_node_alive = dr .ne. r8bad .and. dr .le. drfov_max
								! away from optical axis in radial direction
		    end if

		    if (.not. b_node_alive) then		! Remove node

			nvote = node_cnt(inode,ipool)		! Number of votes in pool

			n_nodes(HTM__NODE_NVOTEMAX,ipool) = max(n_nodes(HTM__NODE_NVOTEMAX,ipool),nvote)

			if (nvote .eq. 0) then			! Safety belt: cannot happen
			    print *, n,ntop,inode,ipool,node_id,node_cnt(inode,ipool),px,py
			    stop 'htm_flush: node with no votes'
			end if

			!print *, 'using',node_id,'with',nvote,' votes'

			i = (inode-1)*max_votes+1
			ngood = smei_htm_combine(iType,bKeepGlitch,iSilent,nvote,node_value(i,ipool),
     &				node_flag(i,ipool),xgood,xbad,badpix)
!     &				node_flag(i,ipool),node_pos(i,ipool),xgood,xbad,badpix)
			if (ngood .gt. 0.0) call smei_htm_node2sky(px,py,xgood,xmg_eq,img_eq,xmg_pl,img_pl)

			i = nint(HTM__CXLO+px*HTM__D_LO)	! Pixel center
			j = nint(HTM__CYLO+py*HTM__D_LO)
			j = min(j,HTM__NYLO)			! Only used if py exactly +90d0

			if (i .lt. 1 .or. i .gt. HTM__NXLO .or. j .lt. 1 .or. j .gt. HTM__NYLO) then
			    write (*,*) ' RA=',px,' Dec=',py	! Better not happen!
			    stop 'htm_flush: bad low res bin, darn'
			end if

			lowres(i,j,HTM__LO_CNT ) = lowres(i,j,HTM__LO_CNT )+1.0
			lowres(i,j,HTM__LO_FRAC) = lowres(i,j,HTM__LO_FRAC)+(nvote-ngood)/float(nvote) ! Dropped nodes as frac of total
			lowres(i,j,HTM__LO_CUT ) = lowres(i,j,HTM__LO_CUT )+xbad ! Dropped adus as fraction of total

			! Decrease counters for nodes and votes

			n_nodes(HTM__NODE_NSLOT,ipool) = n_nodes(HTM__NODE_NSLOT,ipool)-1
			n_nodes(HTM__NODE_NVOTE,ipool) = n_nodes(HTM__NODE_NVOTE,ipool)-nvote

			node_name(inode,ipool) = HTM__NO_NODE	! Necesssary
			node_cnt (inode,ipool) = 0		! Redundant
			node_ra  (inode,ipool) = r8bad		! Redundant
			node_dec (inode,ipool) = r8bad		! Redundant

		    end if

		end if

		n = n+1
		if (n .le. ntop) inode = node_sort(n,ipool)

	    end do

	    ! Update highest occupied index by removing empty slots at top of pool

	    do while (ntop .gt. 0 .and. node_name(ntop,ipool) .eq. HTM__NO_NODE)
		ntop = ntop-1
	    end do

	    if (iSilent .lt. 2) then
		i = 0
		i = i+Int2Str(n_old_nodes(HTM__NODE_NSLOT,ipool)-n_nodes(HTM__NODE_NSLOT,ipool), cStr(i+1:))
		i = i+Str2Str('/'					, cStr(i+1:))
		i = i+Int2Str(n_old_nodes(HTM__NODE_NSLOT,ipool)	, cStr(i+1:))+1
		i = i+Str2Str('nodes,'				, cStr(i+1:))+1
		i = i+Int2Str(n_old_nodes(HTM__NODE_NVOTE,ipool)-n_nodes(HTM__NODE_NVOTE,ipool), cStr(i+1:))
		i = i+Str2Str('/'					, cStr(i+1:))
		i = i+Int2Str(n_old_nodes(HTM__NODE_NVOTE,ipool)	, cStr(i+1:))+1
		i = i+Str2Str('votes,'				, cStr(i+1:))+1
		i = i+Int2Str(n_nodes(HTM__NODE_NTOP,ipool)-ntop	, cStr(i+1:))
		i = i+Str2Str('@end'				, cStr(i+1:))

		call Say('htm_flush','I','pool '//cInt2Str(ipool),cStr)
	    end if

	    n_nodes(HTM__NODE_NTOP,ipool) = ntop

	end if

	return

	entry smei_htm_save()

	call ArrI4Copy(HTM__POOL_S,n_nodes,n_old_nodes)
	n_frame_votes = 0

	return

	entry smei_htm_print()

	i = 0
	i = i+Int2Str(n_frame_votes	, cStr(i+1:))+1
	i = i+Str2Str('votes ('		, cStr(i+1:))
	i = i+Int2Str(n_nodes(HTM__NODE_IEMPTY,1)-n_old_nodes(HTM__NODE_IEMPTY,1), cStr(i+1:))+1
	i = i+Str2Str('in empty slots)'	, cStr(i+1:))

	call Say('htm','I','<hh:mm:ss>',cStr)

	do j=1,HTM__POOL_N
	    i = 0
	    i = i+Int2Str(n_old_nodes(HTM__NODE_NSLOT,j), cStr(i+1:))
	    i = i+Str2Str('/'	, cStr(i+1:))
	    i = i+Int2Str(n_old_nodes(HTM__NODE_NTOP,j), cStr(i+1:))
	    i = i+Str2Str('+'	, cStr(i+1:))
	    i = i+Int2Str(n_nodes(HTM__NODE_NSLOT,j)-n_old_nodes(HTM__NODE_NSLOT,j), cStr(i+1:))
	    i = i+Str2Str('='	, cStr(i+1:))
	    i = i+Int2Str(n_nodes(HTM__NODE_NSLOT,j), cStr(i+1:))
	    i = i+Str2Str('/'	, cStr(i+1:))
	    i = i+Int2Str(n_nodes(HTM__NODE_NTOP,j), cStr(i+1:))
	    i = i+Str2Str(','	, cStr(i+1:))
	    i = i+Int2Str(n_old_nodes(HTM__NODE_NVOTE,j), cStr(i+1:))
	    i = i+Str2Str('+'	, cStr(i+1:))
	    i = i+Int2Str(n_nodes(HTM__NODE_NVOTE,j)-n_old_nodes(HTM__NODE_NVOTE,j), cStr(i+1:))
	    i = i+Str2Str('='	, cStr(i+1:))
	    i = i+Int2Str(n_nodes(HTM__NODE_NVOTE,j), cStr(i+1:))

	    call Say('htm','I','<hh:mm:ss>',cStr)
	end do

	return

	entry smei_htm_print_nodes(mode)

	max_votes = max_votes_mode(mode)
	max_nodes = HTM__VOTES/max_votes

	do j=1,HTM__POOL_N

	    n = n_nodes(HTM__NODE_NTOPMAX,j)

	    i = 0
	    i = i+Int2Str(n				, cStr(i+1:))+1
	    i = i+Str2Str('nodes;'			, cStr(i+1:))+1

	    i = i+Flt2Str(100.0*n/HTM__NODES,2		, cStr(i+1:))
	    i = i+Str2Str('% of node pool;'		, cStr(i+1:))+1

	    i = i+Flt2Str(100.0*n/max_nodes,2		, cStr(i+1:))
	    i = i+Str2Str('% of vote pool#'		, cStr(i+1:))

	    n = n_nodes(HTM__NODE_NVOTEMAX,j)

	    i = i+Int2Str(n				, cStr(i+1:))+1
	    i = i+Str2Str('votes/node;'			, cStr(i+1:))+1

	    i = i+Flt2Str(100.0*n/max_votes,2		, cStr(i+1:))
	    i = i+Str2Str('% of vote/node pool'		, cStr(i+1:))

	    n_nodes(HTM__NODE_NTOPMAX ,j) = 0
	    n_nodes(HTM__NODE_NVOTEMAX,j) = 0

	end do

	call Say('print_nodes','I','used',cStr)

	print *, '????', n_nodes

	return
	end
