//+
// NAME:
//	ftn_wrapper.cpp
// PURPOSE:
//	Collection of C++ routines to access the Spatial Index routines in
//	the JHU HTM package (version 2.0) to be called from Fortran
// CATEGORY:
//	ucsd/camera/for/indexing
// MODIFICATION HISTORY:
//	2002-2003 Aaron Smith (UCSD/CASS)
//	    Spatial Index routines researched
//	    C++ spatial index utilization routines developed.
//-

#include <VarVecDef.h>
#include "VarStr.h"
#include "SpatialIndex.h"
#include "SpatialGeneral.h"
#include "SpatialVector.h"
#include "SpatialConvex.h"

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

#define NORTH 1
#define SOUTH 0

// the extern "C" eliminates all function name mangling, allowing for calls
// to be made from Fortran

extern "C" long smei_htm_pixel_(short *ilevel, float64 ra[], float64 dec[], long *ntr_max, uint64 tr_node[]);
extern "C" void smei_htm_indexpixel_(short *ilevel, float node[],
	uint64 nodeid[], short hits[], float *presponse, float64 ra[], float64 dec[],
	long *pcount, long *ptotal, long *pflag, float *pslope);
extern "C" void trianglecenter_(short *level, uint64 *imagenode, float64 *pra, float64 *pdec);

extern "C" long getindex_(const short level, short idx[]);
extern "C" void getindices_(const short level, const uint64 id, short index[]);

//+
// NAME:
//	smei_htm_pixel
// PURPOSE:
//	Indexes a single pixel
// CATEGORY:
//
// CALLING SEQUENCE:
long smei_htm_pixel_(short *ilevel, float64 ra[], float64 dec[], long *ntr_max, uint64 tr_node[])
// INPUTS:
//	*ilevel		short		indexing level (usually 11)
//	ra[4]		float64		RA of corners of pixel (degrees)
//	dec[4]		float64		declination of corners of pixel (degrees)
// OUTPUTS:
//	*ntr		long		number of triangles to which the pixel contributes
//	tr_node[]	unint64		node id of triangles
// MODIFICATION HISTORY:
//	JAN-2006, Paul Hick (UCSD/CASS)
//-
{
	short		level = *ilevel	;
	long		ntr	;

	short		idx[15]	;			// needs to be at least *ilevel+2 long
	short		triangle;

	float64		*pra	;
	float64		*pdec	;

	int		k	;

	uint64		*partialnode	;
	uint64		*fullnode	;

	SpatialIndex	si(level,2)	;
	SpatialVector	*v[4]		;
	SpatialVector	*tempv		;
	SpatialConvex	*pixelconvex	;
	ValVec<uint64>	*partialvec	;
	ValVec<uint64>	*fullvec	;

 	//initializ dynamic arrays

 	partialvec = new ValVec<uint64>();
 	fullvec    = new ValVec<uint64>();

	//create 4 spatial vectors representing the vertices of the pixel

	v[0] = new SpatialVector(ra[0], dec[0]);
	v[1] = new SpatialVector(ra[1], dec[1]);
	v[2] = new SpatialVector(ra[2], dec[2]);
	v[3] = new SpatialVector(ra[3], dec[3]);

	//create the spatial convex representing the pixel

	pixelconvex = new SpatialConvex(v[0], v[1], v[2], v[3]);

	//free vector memory

	delete v[0];
	delete v[1];
	delete v[2];
	delete v[3];

	//intersect the pixel with the spatial index

	pixelconvex->intersect(&si, partialvec, fullvec);

	//test center of partial nodes to see if they are within pixelconvex

	partialnode = partialvec->vector_;

	//dummy variables for address initialization of pointer variables

	float64 dum1=5, dum2=6;  pra=&dum1; pdec=&dum2;

	//test partial nodes, if the center is in the pixel then keep it

	for (k=0; k<(partialvec->length()); k++, partialnode++)
	{
		si.CASStriangleCenter(*partialnode, pra, pdec );
		tempv = new SpatialVector(*pra, *pdec);

		triangle = pixelconvex->CASStestVertex(*tempv);
		delete tempv;

		// triangle = 1 or 0 for in or out of the pixel
		if (triangle == 1)  fullvec->append(*partialnode);
	}

	delete pixelconvex;
	delete partialvec;

	//give each triangle an equal portion of the response

	fullnode = fullvec->vector_;
	ntr = fullvec->length();

	if (ntr <= *ntr_max)
	{
		for (k=0; k<ntr; k++, fullnode++)
		{
			getindices_(level,*fullnode,idx);
			tr_node[k] = *fullnode;
		}
	}

	delete fullvec;

	return ntr;
}

//+
// NAME:
//	smei_htm_indexpixel
// PURPOSE:
//	Indexes a single pixel
// CATEGORY:
//
// CALLING SEQUENCE:
void smei_htm_indexpixel_(short *ilevel, float node[], uint64 nodeid[], short hits[],
	float *presponse, float64 ra[], float64 dec[], long *pcount, long *ptotal,
	long *pflag, float *pslope)
// INPUTS:
//	*ilevel		short		indexing level (usually 11)
//	node[]		float
//	nodeid[]	unint64
//	hits[]		short
//	*presponse	float
//	ra[4]		float64		RA of corners of pixel (degrees)
//	dec[4]		float64		declination of corners of pixel (degrees)
//	*pflag		long		bitwise set of flags to control behavior of routine:
//					bit 0:
//					bit 1:
//					bit 2:
//					bit 3:
//	*pslope		float
// OUTPUTS:
//	hits		short		updated
//	*pcount		long
//					if *piflag = 2 then *pcount = 0
//	*ptotal		long		number of triangles to which pixel contributes
// CALLS:
//	getindices_, getindex_
// INCLUDE:
// SIDE EFFECTS:
// RESTRICTIONS:
// PROCEDURE:
//	Aarons original version of the indexing program and this routine
//	had two flags iflag (passed as piflag into this routine) and mflag
//	(passes as pmflag).
//
//	The original indexing program had iflag=0 for cameras 1 and 2; iflag=1
//	would happen for certain pixels (hot/flipper pixels?). However, the
//	original version of this routine did not test for iflag=0 or iflag=1
//	(only for iflag=2).
//
//	The original indexing program set mflag=0 or 1, where mflag=1
//	indicated a suspected measle (incl. 'white measle'). This flag was
//	the main active ingredient in the original version of this routine.
//	(sometimes iflag = 1 would result in mflag = 1, so iflag=1 would
//	impact this routine only indirectly).
//
//	Now iflag=2 is available by setting bit 2 of pflag (add 4;
//	this sets itime=1 in this routine).
//	[CURRENTLY THIS SECTION IS COMMENTED OUT]
//
//	mflag=0,1 is available now by setting bit 0 of pflag. This will set ihot
//	in this routine (so ihot is the old mflag).
//
//	In addition setting bit 1 of pflag (add 2; this sets ifull in this routine)
//	will unconditionally drop the full response in each relevant triangle.
//	This is not used anywhere.
// MODIFICATION HISTORY:
//	2002-2003, Aaron Smith
//	DEC-2004, Paul Hick (UCSD/CASS)
//	    Made idx[] a fixed length array of 15 elements (PGI compiler doesn't
//	    support variable length arrays). This is enough for level 13 (the highest
//	    level that works with the number of nodes within long integer range).
//	    Changed input args count, total from uint64 to long; combined args piflag,
//	    pmflag into single pflag.
//	MAR-2005, Paul Hick, Aaron Smith (UCSD/CASS)
//		Added argument pslope.
//	JAN-2006, Paul Hick (UCSD/CASS)
//	    Removed two redundant lines:
//		nodeid[index] = *fullnode;
//-
{
	short		level    = *ilevel	;
	float		response = *presponse	;

	long		ihot  = (*pflag & 1)/1	;	// Bit 0 set
	long		ifull = (*pflag & 2)/2	;	// Bit 1 set
	long		itime = (*pflag & 4)/4	;	// Bit 2 set

	float		slope = *pslope ;

	short		idx[15]	;			// needs to be at least *ilevel+2 long
	long		tcount	;
	long		ttotal	;
	short		triangle;

	size_t		flen	;
	long		index	;
	int		k	;

	float64		*pra		;
	float64		*pdec		;
	float64		ratio		;
	float64		diff		;
	uint64		*partialnode	;
	uint64		*fullnode	;

	SpatialIndex	si(level,2)	;
	SpatialVector	*v[4]		;
	SpatialVector	*tempv		;
	SpatialConvex	*pixelconvex	;
	ValVec<uint64>	*partialvec	;
	ValVec<uint64>	*fullvec	;

 	//initializ dynamic arrays

 	partialvec = new ValVec<uint64>();
 	fullvec    = new ValVec<uint64>();

	//create 4 spatial vectors representing the vertices of the pixel

	v[0] = new SpatialVector(ra[0], dec[0]);
	v[1] = new SpatialVector(ra[1], dec[1]);
	v[2] = new SpatialVector(ra[2], dec[2]);
	v[3] = new SpatialVector(ra[3], dec[3]);

	//create the spatial convex representing the pixel

	pixelconvex = new SpatialConvex(v[0], v[1], v[2], v[3]);

	//free vector memory

	delete v[0];
	delete v[1];
	delete v[2];
	delete v[3];

	//intersect the pixel with the spatial index

	pixelconvex->intersect(&si, partialvec, fullvec);

	//test center of partial nodes to see if they are within pixelconvex

	partialnode = partialvec->vector_;

	//dummy variables for address initialization of pointer variables

	float64 dum1=5, dum2=6;  pra=&dum1; pdec=&dum2;

	//test partial nodes, if the center is in the pixel then keep it

	for (k=0; k<(partialvec->length()); k++, partialnode++)
	{
		si.CASStriangleCenter(*partialnode, pra, pdec );
        	tempv = new SpatialVector(*pra, *pdec);

		triangle = pixelconvex->CASStestVertex(*tempv);
        	delete tempv;

 	       // triangle = 1 or 0 for in or out of the pixel
        	if (triangle == 1)  fullvec->append(*partialnode);
	}

	delete pixelconvex;
	delete partialvec;

	//give each triangle an equal portion of the response

	fullnode = fullvec->vector_;
	flen   = fullvec->length();


	tcount = 0;
	ttotal = 0;

	for (k=0; k<flen; k++, fullnode++)
	{
		getindices_(level,*fullnode, idx);
		index = getindex_(level,idx);

		nodeid[index] = *fullnode;

		ttotal++;

		if (ifull == 1)				// Drop full response in each triangle
		{
			node  [index] += response;
			hits  [index]++;
		}
/*
		else if (itime == 1)		// timing map making section ??
		{
			if (hits[index] == 0)
			{
				node[index] += response;
				hits[index]++;
			}
			else
			{
				ratio = response/(node[index]/hits[index]);
				if (ratio < 0.0)
				{
					node[index] = node[index]-response;
					hits[index]++;
				}
				else if (ratio > 30.0)
				{
					node[index] = -node[index]-response;
					hits[index]++;
				}
				else
				{
					node[index] += response;
					hits[index]++;
				}
			}
		}
*/
		else
		{
			// Algorithm to eliminate bad pixels, which are usually brighter than what
			// they should be, but when flagged, they could be dimmer (downflippers or
			// "white measles")

			// First hit, put it in, unless flagged as possible downflipper
			// Second hit, add it in: unless > 1.33x of first,
			// or --if flagged-- unless < 0.75x of first further hits, add it in:
			// unless > 1.25x of prev. avg., or --if flagged-- unless < 0.8x of prev. avg.

			// The addition of a slope parameter (used only near bright stars)
			// eliminates nearly all star nibbling if the slope is high

			if (hits[index] > 1)			// Multiple hits section
			{
				ratio = response/(node[index]/hits[index]);
				diff  = response-(node[index]/hits[index]);
				if (slope > 50.0 )		// In a star, check for a crazy ratio
				{
					if (ratio > 3.3)	// Response is larger than node
					{		
						tcount++;
					}
					else if (ihot == 1 && ratio < 0.1 && diff < -50.0)
					{
						tcount++;
					}
					else
					{
						node[index] += response;
						hits[index]++;
					}
				}
				else
				{
					if (ratio > 1.25 && diff > 50.0)
					{			// response is larger than node
						tcount++;	// Cosmic ray ??
					}
					else if (ihot == 1 && ratio < 0.8 && diff < -50.0)
					{
						tcount++;	// Confirmed downflipper ??
					}
					else			// OK
					{
						node[index] += response;
						hits[index]++;
					}
				}
			}
			else if (hits[index] == 1)		// Single hits section
			{
				ratio = response/node[index];
				diff  = response-node[index];

				if (slope > 50.0)		// In a star, check for a crazy ratio
				{
					if (ratio > 3.3)	// response is larger than node
					{
						tcount++;
					}
					else if (ihot == 1 && ratio < 0.1 && diff < -50.0)
					{
						tcount++;
					}
					else
					{
						node[index] += response;
						hits[index]++;
					}
				}
				else
				{
					if (ratio > 1.333 && diff > 50.0)// response is larger than node
					{
						tcount++;		// Cosmic ray
					}
					else if (ihot == 1 && ratio < 0.75 && diff < -50.0)
					{
						tcount++;		// Confirmed downflipper
					}
					else
					{
						node[index] += response;// OK
						hits[index]++;
					}
				}
			}
			else 						// First hit section, hits[index] = 0
			{
				if (ihot == 1)
				{
					tcount++;
				}
				else
				{
					node  [index] = response;
					hits  [index] = 1;
				}
			}
		}
	}

	*pcount = tcount;
	*ptotal = ttotal;

	delete fullvec;

	return;
}
//+
// NAME:
//	getindices_
// PURPOSE:
//	Get the indices(0-3) of an IDTYPE nodename
//	the first indice(N or S) = 1 or 2
//-
void getindices_(const short level, const uint64 id, short index[])
{
	uint32	size = 0;
	uint32	i;
	bool	ok = true;

	for (i = 0; i < level+2; i++)
	{
		index[i] = 0;
	}
  
	//working with the same level i should be pre-determined

	for (i = 0; i < IDSIZE; i += 2)
	{
		if( (id << i) & IDHIGHBIT ) break;
		if( (id << i) & IDHIGHBIT2)
		{
			ok = false;	//invalid id
			break;
		}
	}
  
	if (!ok || id == 0)
	{
		return;			//invalid id->exit
	}

	size = (IDSIZE-i) >> 1;

	for (i=0; i < size-1; i++)
	{
		index[size-i-1] = ( (id >> i*2) & 3);
	}

	if ( (id>>(size*2-2)) & 1)
	{
		index[0] = NORTH;
	}
	else
	{
		index[0] = SOUTH;
	}

	return;
}
//+
// NAME:
//	getindex_
// PURPOSE:
//	convert index_size(level+2) indices to one indice
//-
long getindex_(const short level, short idx[])
{
	int	i	;
	int	e	;
	int	p	;
	long	max	;
	long	indice = 0;

	for (i = 0, e = level+1; i < level+2; i++, e--)
	{
		p = pow(4,e);
		if (e == 0) p = 1;
		indice += idx[i]*p;
	}

	//for (i = 0, max=2; i <= level; i++) { max = 4*max }
	max = 2*pow(4,level+1);

	if (indice > max || indice < 0)
	{
		printf("Bad Indice: %ld\n",indice);
		printf("Indices: ");
		for (i = 0; i < level+2; i++)
		{
			printf("%d, ",idx[i]);
		}
		exit(1);
  	}

	return indice;
}
//+
// NAME:
//	trianglecenter_
// PURPOSE:
//	routine to find the center ra/dec of a given spatial triangle
//	note: when called from fortran, this function is faster than
//	a simple wrapper for CASStrianglecenter(spatialIndex.cpp and called
//	in indexpixel_ function)
//-
void trianglecenter_(short *level, uint64 * imagenode, float64 * pra, float64 * pdec)
{
	SpatialIndex	si(*level,2);
	SpatialVector	v1;
	SpatialVector	v2;
	SpatialVector	v3;

	//get vertices of triangle

	si.nodeVertex(*imagenode, v1, v2, v3);

	//average over ra/dec of vertices to find center ra/dec

	*pra  = ( v1.ra () + v2.ra () + v3.ra () ) / 3;
	*pdec = ( v1.dec() + v2.dec() + v3.dec() ) / 3;

	if ( (fabs(*pra-v1.ra())) > 10 )
	{
		if (v1.ra() > 180)  v1.set(v1.ra()-360,v1.dec());
		if (v2.ra() > 180)  v2.set(v2.ra()-360,v2.dec());
		if (v3.ra() > 180)  v3.set(v3.ra()-360,v3.dec());

		*pra = ( v1.ra() + v2.ra() + v3.ra() ) / 3;
		// 0<ra<360
		if (*pra < 0) *pra = *pra+360;
	}

	// These don't work because destructor needs pointer type.
	// I'm not even sure these are necessary...
	//delete v1;  delete v2;  delete v3;

	return;
}
