/************************************************************************************************
C+
C NAME:
C	A_C_routines.cpp
C PURPOSE:
C	This is a collection of c++ routines that access the JHU Spatial Index routines for
C	SMEI Data analysis.  All routines are to be called from Fortran.
C CATEGORY:
C	SMEI Data Analysis
C MODIFICATION HISTORY:
C	2002-2003 Aaron Smith (UCSD/CASS)
C		Spatial Index routines researched, C++ spatial index utilization routines developed.
C-
**********************************************************************************************/
#include <VarVecDef.h>
#include "VarStr.h"
#include "SpatialIndex.h"
#include "SpatialGeneral.h"
#include "SpatialVector.h"
#include "SpatialConvex.h"

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

#define NORTH 1
#define SOUTH 0

//the extern "C" eliminates all function name mangling, allowing for calls
//to be made from Fortran
extern "C" void indexpixel_(short * ilevel, float node[], uint64 nodeid[], short hits[],float64 * pixelresponse,
		float64 ra[], float64 dec[], uint64 * count, uint64 * total, short * piflag, short * pmflag,
		float *ap, float *bp, float * pslope);
extern "C" void indexpixelnoalg_(short * ilevel, float node[], uint64 nodeid[], short hits[],
		float64 * pixelresponse, float64 ra[], float64 dec[], uint64 * count);
extern "C" void trianglecenter_(short *level, uint64 *imagenode, float64 * pra, float64 * pdec);
extern "C" long getindex_(const short level, short idx[]);
extern "C" void getindices_(const short level, const uint64 id, short index[]);
//////////////////////////////////////////////////////////////////
//	routine to index a pixel, with CRX algorithm
//////////////////////////////////////////////////////////////////
void indexpixel_(short * ilevel, float node[], uint64 nodeid[], short hits[],float64 * pixelresponse,
		float64 ra[], float64 dec[], uint64 * count, uint64 * total, short * piflag, short * pmflag,
		float * ap, float * bp, float * pslope)
{
  short level=*ilevel;
  size_t flen;
  long index;
  int i, j, k, a, b, p, n, f,iflag=*piflag,mflag=*pmflag;
  float64 *pra, *pdec, stddev, stddev2, discrep, ratio, diff;
  float response=*pixelresponse;
  float as=*ap,bs=*bp, slope=*pslope;
  SpatialIndex si(level,2);
  SpatialVector *v[4], *tempv;
  SpatialConvex *pixelconvex;
  ValVec<uint64> *partialvec, *fullvec, *tempfullvec;
  uint64 *partialnode, *fullnode, imagenode, tcount, ttotal;
  short idx[level+2], triangle;

     //initializ dynamic arrays
     partialvec = new ValVec<uint64>();
     fullvec = new ValVec<uint64>();

     //create 4 spatial vectors representing the vertices of the pixel
     v[0] = new SpatialVector(ra[0], dec[0]);
     v[1] = new SpatialVector(ra[1], dec[1]);
     v[2] = new SpatialVector(ra[2], dec[2]);
     v[3] = new SpatialVector(ra[3], dec[3]);

     //create the spatial convex representing the pixel
     pixelconvex = new SpatialConvex(v[0], v[1], v[2], v[3]);

     //free vector memory
     delete v[0];
     delete v[1];
     delete v[2];
     delete v[3];

     //intersect the pixel with the spatial index
     pixelconvex->intersect(&si, partialvec, fullvec);

     //test center of partial nodes to see if they are within pixelconvex
     partialnode = partialvec->vector_;

     //dummy variables for address initialization of pointer variables
     float64 dum1=5, dum2=6;  pra=&dum1; pdec=&dum2;

     //test partial nodes, if the center is in the pixel then keep it
     for (k=0; k<(partialvec->length()); k++, partialnode++)
     {
	si.CASStriangleCenter(*partialnode, pra, pdec );
        tempv = new SpatialVector(*pra, *pdec);

	triangle = pixelconvex->CASStestVertex(*tempv);
        delete tempv;

        // triangle = 1 or 0 for in or out of the pixel
        if (triangle == 1)  fullvec->append(*partialnode);
     }

     delete pixelconvex;
     delete partialvec;

     //give each triangle an equal portion of the response
     fullnode = fullvec->vector_;
     flen = fullvec->length();
     tcount=0;
     ttotal=0;
     
     for(k=0; k<flen; k++, fullnode++)
     {
        getindices_(level,*fullnode, idx);
	index = getindex_(level,idx);

	nodeid[index] = *fullnode;
//	node[index]+=response;
//	hits[index]++;
//	if(iflag<2)
//	{
//	  printf("this is iflag %d\n",iflag);
//	  printf("and response %f\n",response);
//	}
	ttotal++;
	if(iflag==2)				//timing map making section
	{
	  if(hits[index]==0)
	  {
	    node[index]+=response;
	    hits[index]++;
	  }else
	  {
	    ratio=response/(node[index]/hits[index]);
	    if(ratio<0.)
	    {
	      node[index]=node[index]-response;
	      hits[index]++;
	    }
	    else if(ratio>30.)
	    {
	      node[index]=-node[index]-response;
	      hits[index]++;
	    }
	    else
	    {
	    node[index]+=response;
	    hits[index]++;
	    }
	  }
	}else
	{

	//algorithm to eliminate bad pixels, which are assumed usually to be brighter
	//than what they should be, but when flagged, they could be dimmer (downflippers or "white measles")
//
//	First hit, put it in, unless flagged as possible downflipper
//	Second hit, add it in: unless > 1.33x of first, or --if flagged-- unless < 0.75x of first
//	further hits, add it in: unless > 1.25x of prev. avg., or --if flagged-- unless < 0.8x of prev. avg.
//
//	The addition of a slope parameter (used only near bright stars) eliminates nearly all star nibbling
//	If the slope is high
//
	  if(hits[index]>1)	//multiple hits section
	  {
	    ratio=response/(node[index]/hits[index]);
	    diff=response-(node[index]/hits[index]);
	    if(slope>50.0)
	    {
	     //In a star, check for a crazy ratio
	     if(ratio > 3.3)
	     {
	      //response is larger than node
	      tcount++;
	     }else if(ratio < 0.1 && mflag == 1 && diff<-50.)
	     {
	      tcount++;
	     }else
	     {
	      node[index]+=response;
	      hits[index]++;
	     }
	    }
	    else
	    {
	     if(ratio > 1.25 && diff>50.)
	     {
	      //response is larger than node
	      tcount++;
//	        printf("	Ra/Dec: [%f,%f]\n",ra[1],dec[1]);
//	        printf("	 Response, Ratio, Diff: %f, %f , %f\n",response,ratio,diff);
//		printf("	  Slope a,b: %f, %f, %f\n",slope,as,bs);
	     }else if(ratio < 0.8 && mflag == 1 && diff<-50.)
	     {
	      tcount++;
	     }else
	     {
	      node[index]+=response;
	      hits[index]++;
	     }
	    }
	  }else if(hits[index]==1)	//single hits section
	  {
	    ratio=response/node[index];
	    diff=response-node[index];
	    if(slope>50.0)
	    {
	     //In a star, check for a crazy ratio
	     if(ratio > 3.3)
	     {
	      //response is larger than node
	      tcount++;
	     }else if(ratio < 0.1 && mflag == 1 && diff<-50.)
	     {
	      tcount++;
	     }else
	     {
	      node[index]+=response;
	      hits[index]++;
	     }
	    }
	    else
	    {
	     if(ratio > 1.333 && diff>50.)
	     {
	      //response is larger than node
	      tcount++;
	     }else if(ratio < 0.75 && mflag == 1 && diff<-50.)
	     {
	      tcount++;
	     }else
	     {
	      node[index]+=response;
	      hits[index]++;
	     }
	    }
	  }else 			//first hit section
	  {
	      if(mflag==0)
	      {
	      node[index]=response;
	      hits[index]=1;
	      nodeid[index]=*fullnode;
	      }else
	      {
	          tcount++;
	      }
	  }
	}
     }
     *count=tcount;
     *total=ttotal;

     delete fullvec;
     return;
}
//////////////////////////////////////////////////////////////////
//	routine to index a pixel
//////////////////////////////////////////////////////////////////
void indexpixelnoalg_(short * ilevel, float node[], uint64 nodeid[], short hits[],
		float64 * pixelresponse, float64 ra[], float64 dec[], uint64 * count)
{
  short level=*ilevel;
  size_t flen;
  long index;
  int i, j, k, a, b, p, n, f;
  float64 *pra, *pdec, stddev, stddev2, discrep, ratio;
  float response=*pixelresponse;
  SpatialIndex si(level,2);
  SpatialVector *v[4], *tempv;
  SpatialConvex *pixelconvex;
  ValVec<uint64> *partialvec, *fullvec, *tempfullvec;
  uint64 *partialnode, *fullnode, imagenode, tcount;
  short idx[level+2], triangle;

     //initializ dynamic arrays
     partialvec = new ValVec<uint64>();
     fullvec = new ValVec<uint64>();

     //create 4 spatial vectors representing the vertices of the pixel
     v[0] = new SpatialVector(ra[0], dec[0]);
     v[1] = new SpatialVector(ra[1], dec[1]);
     v[2] = new SpatialVector(ra[2], dec[2]);
     v[3] = new SpatialVector(ra[3], dec[3]);

     //create the spatial convex representing the pixel
     pixelconvex = new SpatialConvex(v[0], v[1], v[2], v[3]);

     //free vector memory
     delete v[0];
     delete v[1];
     delete v[2];
     delete v[3];

     //intersect the pixel with the spatial index
     pixelconvex->intersect(&si, partialvec, fullvec);

     //test center of partial nodes to see if they are within pixelconvex
     partialnode = partialvec->vector_;

     //dummy variables for address initialization of pointer variables
     float64 dum1=5, dum2=6;  pra=&dum1; pdec=&dum2;

     //test partial nodes, if the center is in the pixel then keep it
     for (k=0; k<(partialvec->length()); k++, partialnode++)
     {
	si.CASStriangleCenter(*partialnode, pra, pdec );
        tempv = new SpatialVector(*pra, *pdec);

	triangle = pixelconvex->CASStestVertex(*tempv);
        delete tempv;

        // triangle = 1 or 0 for in or out of the pixel
        if (triangle == 1)  fullvec->append(*partialnode);
     }

     delete pixelconvex;
     delete partialvec;

     //give each triangle an equal portion of the response
     fullnode = fullvec->vector_;
     flen = fullvec->length();

     tcount=0;
     for(k=0; k<flen; k++, fullnode++)
     {
        getindices_(level,*fullnode, idx);
	index = getindex_(level,idx);

	  nodeid[index] = *fullnode;
	  node[index]+=response;
	  hits[index]++;
     }

     delete fullvec;
     return;
}
//////////////////////////////////////////////////////////////////
//	Get the indices(0-3) of an IDTYPE nodename
//	the first indice(N or S) = 1 or 2
//////////////////////////////////////////////////////////////////
void getindices_(const short level, const uint64 id, short index[])
{
  uint32 size=0, i;
  bool ok=true;

  for(i=0; i<level+2; i++)
  {
    index[i] = 0;
  }
  
  //working with the same level i should be pre-determined
  for(i=0; i<IDSIZE; i+=2)
  {
    if( (id<<i) & IDHIGHBIT) break;
    if( (id<<i) & IDHIGHBIT2)
    {
      ok=false;//invalid id
      break;
    }
  }
  
  if(!ok || id==0)
  {
    return;//invalid id->exit
  }

  size=(IDSIZE-i) >> 1;

  for(i=0; i<size-1; i++)
  {
    index[size-i-1] = ( (id >> i*2) & 3);
  }

  if( (id>>(size*2-2)) & 1)
  {
    index[0] = NORTH;
  }else
  {
    index[0] = SOUTH;
  }

  return;
}
//////////////////////////////////////////////////////////////////
//	convert index_size(level+2) indices to one indice
//////////////////////////////////////////////////////////////////
long getindex_(const short level, short idx[])
{
  int i,e,p;
  long indice=0, max=0;

  max = 2*pow(4,level+1);

  for(i=0,e=level+1; i<level+2; i++,e--)
  {
    p=pow(4,e);
    if(e==0) p=1;
    indice += idx[i]*p;
  }
  if(indice>max || indice<0)
  {
    printf("Bad Indice: %ld\n",indice);
    printf("Indices: ");
    for(i=0; i<level+2; i++)
    {
      printf("%d, ",idx[i]);
    }
    exit(1);
  }

  return indice;
}
//////////////////////////////////////////////////////////////////////////////
//	routine to find the center ra/dec of a given spatial triangle
//	note: when called from fortran, this function is faster than
//	  a simple wrapper for CASStrianglecenter(spatialIndex.cpp and called
//	  in indexpixel_ function)
//////////////////////////////////////////////////////////////////////////////
void trianglecenter_(short * level, uint64 * imagenode, float64 * pra, float64 * pdec)
{
  SpatialIndex si(*level,2);
  SpatialVector v1,v2,v3;

  //get vertices of triangle
  si.nodeVertex(*imagenode, v1, v2, v3);

  //average over ra/dec of vertices to find center ra/dec
  *pra = ( v1.ra() + v2.ra() + v3.ra() ) / 3;
  *pdec = ( v1.dec() + v2.dec() + v3.dec() ) / 3;

  if ( (fabs(*pra - v1.ra())) > 10 )
  {
    if (v1.ra()>180)  v1.set(v1.ra() - 360,v1.dec());
    if (v2.ra()>180)  v2.set(v2.ra() - 360,v2.dec());
    if (v3.ra()>180)  v3.set(v3.ra() - 360,v3.dec());

  *pra = ( v1.ra() + v2.ra() + v3.ra() ) / 3;
  // 0<ra<360
  if(*pra<0)  *pra = *pra + 360;
  }

  // These don't work because destructor needs pointer type.
  // I'm not even sure these are necessary...
  //delete v1;  delete v2;  delete v3;

  return;
}



